/*
 * MCS Media Computer Software Copyright (c) 2006 by MCS
 * -------------------------------------- Created on 06.01.2006 by w.klaas
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package de.mcs.utils;

/**
 * This class encapsulates methods to deal with arrays of numbers, converting
 * from numbers to bytes and bytes to numbers.
 * <p>
 * Methods xxxToByte() convert a Java array of primitive numbers (int, short,
 * ...) to a Java array of bytes. Methods byteToXxx() convert from a Java array
 * of bytes into a Java array of primitive numbers (int, short, ...)
 * <p>
 * Variant interfaces convert a section of an array, and also can convert to
 * sub-classes of Java <b>Number</b>.
 * <P>
 */
public final class Conversions {

  /** to prevent instancing. */
  private Conversions() {

  }

  /**
   * Convert an array of bytes into an array of ints.
   * 
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of int
   */
  public static int[] byteToInt(final byte[] data) throws ConversionException {
    if ((data.length % 4) > 0) {
      throw new ConversionException("array length doesn't match.");
    }
    int[] values = new int[data.length >> 2];

    int i = 0;
    for (byte b : data) {
      int pos = i >> 2;
      values[pos] <<= 8;
      int aByte = b < 0 ? b + 256 : b;
      values[pos] |= aByte;
      i++;
    }
    return values;
  }

  /**
   * Convert an array of bytes into an array of floats.
   * 
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of float
   */
  public static float[] byteToFloat(final byte[] data) throws ConversionException {
    throw new ConversionException("not implemented yet.");
  }

  /**
   * Convert an array of bytes into an array of shorts.
   * 
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of short
   */
  public static short[] byteToShort(final byte[] data) throws ConversionException {
    if ((data.length % 2) > 0) {
      throw new ConversionException("array length doesn't match.");
    }
    short[] values = new short[data.length >> 1];

    int i = 0;
    for (byte b : data) {
      int pos = i >> 1;
      values[pos] <<= 8;
      int aByte = b < 0 ? b + 256 : b;
      values[pos] |= aByte;
      i++;
    }
    return values;
  }

  /**
   * Convert an array of bytes into an array of long.
   * 
   * @param data
   *            The input array of bytes
   * 
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of long
   * 
   */
  public static long[] byteToLong(final byte[] data) throws ConversionException {
    if ((data.length % 8) > 0) {
      throw new ConversionException("array length doesn't match.");
    }
    long[] values = new long[data.length >> 3];

    int i = 0;
    for (byte b : data) {
      int pos = i >> 3;
      values[pos] <<= 8;
      int aByte = b < 0 ? b + 256 : b;
      values[pos] |= aByte;
      i++;
    }
    return values;
  }

  /**
   * Convert an array of bytes into an array of double.
   * 
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of double
   */
  public static double[] byteToDouble(final byte[] data) throws ConversionException {
    throw new ConversionException("not implemented yet.");
  }

  /**
   * Convert a range from an array of bytes into an array of int.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param len
   *            The number of 'int' to convert
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of 'len' int
   */
  public static int[] byteToInt(final int start, final int len, final byte[] data) throws ConversionException {
    byte[] datas = new byte[len];
    System.arraycopy(data, start, datas, 0, len);
    return byteToInt(datas);
  }

  /**
   * Convert 4 bytes from an array of bytes into a single int.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return The integer value of the bytes.
   */
  public static int byteToInt(final byte[] data, final int start) throws ConversionException {
    int[] ival = new int[1];
    ival = byteToInt(start, 4, data);
    return (ival[0]);
  }

  /**
   * Convert a range from an array of bytes into an array of short.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param len
   *            The number of 'short' to convert
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of 'len' short
   */
  public static short[] byteToShort(final int start, final int len, final byte[] data) throws ConversionException {
    byte[] datas = new byte[len];
    System.arraycopy(data, start, datas, 0, len);
    return byteToShort(datas);
  }

  /**
   * Convert 2 bytes from an array of bytes into a single short.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return The short value of the bytes.
   */
  public static short byteToShort(final byte[] data, final int start) throws ConversionException {
    short[] sval = new short[1];
    sval = byteToShort(start, 2, data);
    return (sval[0]);
  }

  /**
   * Convert a range from an array of bytes into an array of float.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param len
   *            The number of 'float' to convert
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of 'len' float
   */
  public static float[] byteToFloat(final int start, final int len, final byte[] data) throws ConversionException {
    byte[] datas = new byte[len];
    System.arraycopy(data, start, datas, 0, len);
    return byteToFloat(datas);
  }

  /**
   * Convert 4 bytes from an array of bytes into a single float.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return The float value of the bytes.
   */
  public static float byteToFloat(final byte[] data, final int start) throws ConversionException {
    float[] fval = new float[1];
    fval = byteToFloat(start, 4, data);
    return (fval[0]);
  }

  /**
   * Convert a range from an array of bytes into an array of long.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param len
   *            The number of 'long' to convert
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of 'len' long
   */
  public static long[] byteToLong(final int start, final int len, final byte[] data) throws ConversionException {
    byte[] datas = new byte[len];
    System.arraycopy(data, start, datas, 0, len);
    return byteToLong(datas);
  }

  /**
   * Convert 8(?) bytes from an array of bytes into a single long.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return The long value of the bytes.
   */
  public static long byteToLong(final byte[] data, final int start) throws ConversionException {
    long[] lval = new long[1];
    lval = byteToLong(start, 8, data);
    return (lval[0]);
  }

  /**
   * Convert a range from an array of bytes into an array of double.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param len
   *            The number of 'double' to convert
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of 'len' double
   */
  public static double[] byteToDouble(final int start, final int len, final byte[] data) throws ConversionException {
    byte[] datas = new byte[len];
    System.arraycopy(data, start, datas, 0, len);
    return byteToDouble(datas);
  }

  /**
   * Convert 8 bytes from an array of bytes into a single double.
   * 
   * @param start
   *            The position in the input array of bytes to start
   * @param data
   *            The input array of bytes
   * @throws ConversionException
   *             if something goes wrong.
   * @return The double value of the bytes.
   */
  public static double byteToDouble(final byte[] data, final int start) throws ConversionException {
    double[] dval = new double[1];
    dval = byteToDouble(start, 8, data);
    return (dval[0]);
  }

  /**
   * Convert a range from an array of int into an array of bytes.
   * 
   * @param start
   *            The position in the input array of int to start
   * @param len
   *            The number of 'int' to convert
   * @param data
   *            The input array of int
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of bytes
   */
  public static byte[] intToByte(final int start, final int len, final int[] data) throws ConversionException {
    int[] values = new int[len];
    byte[] result = new byte[len * 4];
    System.arraycopy(data, start, values, 0, len);
    for (int i = 0; i < values.length; i++) {
      byte[] news = intToByte(values[i]);
      System.arraycopy(news, 0, result, i * 4, news.length);
    }
    return result;
  }

  /**
   * Convert a range from an array of short into an array of bytes.
   * 
   * @param start
   *            The position in the input array of int to start
   * @param len
   *            The number of 'short' to convert
   * @param data
   *            The input array of short
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of bytes
   */
  public static byte[] shortToByte(final int start, final int len, final short[] data) throws ConversionException {
    short[] values = new short[len];
    byte[] result = new byte[len * 2];
    System.arraycopy(data, start, values, 0, len);
    for (int i = 0; i < values.length; i++) {
      byte[] news = shortToByte(values[i]);
      System.arraycopy(news, 0, result, i * 2, news.length);
    }
    return result;
  }

  /**
   * Convert a range from an array of float into an array of bytes.
   * 
   * @param start
   *            The position in the input array of int to start
   * @param len
   *            The number of 'float' to convert
   * @param data
   *            The input array of float
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of bytes
   */
  public static byte[] floatToByte(final int start, final int len, final float[] data) throws ConversionException {
    throw new ConversionException("not implemented yet.");
  }

  /**
   * Convert a range from an array of long into an array of bytes.
   * 
   * @param start
   *            The position in the input array of int to start
   * @param len
   *            The number of 'long' to convert
   * @param data
   *            The input array of long
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of bytes
   */
  public static byte[] longToByte(final int start, final int len, final long[] data) throws ConversionException {
    long[] values = new long[len];
    byte[] result = new byte[len * 8];
    System.arraycopy(data, start, values, 0, len);
    for (int i = 0; i < values.length; i++) {
      byte[] news = longToByte(values[i]);
      System.arraycopy(news, 0, result, i * 8, news.length);
    }
    return result;
  }

  /**
   * Convert a range from an array of double into an array of bytes.
   * 
   * @param start
   *            The position in the input array of double to start
   * @param len
   *            The number of 'double' to convert
   * @param data
   *            The input array of double
   * @throws ConversionException
   *             if something goes wrong.
   * @return an array of bytes
   */
  public static byte[] doubleToByte(final int start, final int len, final double[] data) throws ConversionException {
    throw new ConversionException("not implemented yet.");
  }

  /**
   * Convert a single byte into an array of one byte.
   * <p>
   * (This is a trivial method.)
   * 
   * @param data
   *            The input byte
   * @return an array of bytes
   */
  public static byte[] byteToByte(final byte data) {
    return new byte[] { data };
  }

  /**
   * Convert a single Byte object into an array of one byte.
   * <p>
   * (This is an almost trivial method.)
   * 
   * @param data
   *            The input Byte
   * @return an array of bytes
   */
  public static byte[] byteToByte(final Byte data) {
    return byteToByte(data.byteValue());
  }

  /**
   * Convert a single int into an array of 4 bytes.
   * 
   * @param data
   *            The input int
   * @return an array of bytes
   */
  public static byte[] intToByte(final int data) {
    byte[] value = new byte[4];
    value[0] = (byte) (data >> 24);
    value[1] = (byte) (data >> 16);
    value[2] = (byte) (data >> 8);
    value[3] = (byte) (data);
    return value;
  }

  /**
   * Convert a single Integer object into an array of 4 bytes.
   * 
   * @param data
   *            The input Integer
   * @return an array of bytes
   */
  public static byte[] intToByte(final Integer data) {
    return intToByte(data.intValue());
  }

  /**
   * Convert a single short into an array of 2 bytes.
   * 
   * @param data
   *            The input short
   * @return an array of bytes
   */
  public static byte[] shortToByte(final short data) {
    byte[] value = new byte[2];
    value[0] = (byte) (data >> 8);
    value[1] = (byte) (data);
    return value;
  }

  /**
   * Convert a single Short object into an array of 2 bytes.
   * 
   * @param data
   *            The input Short
   * @return an array of bytes
   */
  public static byte[] shortToByte(final Short data) {
    return shortToByte(data.shortValue());
  }

  /**
   * Convert a single float into an array of 4 bytes.
   * 
   * @param data
   *            The input float
   * @return an array of bytes
   */
  public static byte[] floatToByte(final float data) {
    return null;
  }

  /**
   * Convert a single Float object into an array of 4 bytes.
   * 
   * @param data
   *            The input Float
   * @return an array of bytes
   */
  public static byte[] floatToByte(final Float data) {
    return floatToByte(data.floatValue());
  };

  /**
   * Convert a single long into an array of 8 bytes.
   * 
   * @param data
   *            The input long
   * @return an array of bytes
   */
  public static byte[] longToByte(final long data) {
    byte[] value = new byte[8];
    value[0] = (byte) (data >> 56);
    value[1] = (byte) (data >> 48);
    value[2] = (byte) (data >> 40);
    value[3] = (byte) (data >> 32);
    value[4] = (byte) (data >> 24);
    value[5] = (byte) (data >> 16);
    value[6] = (byte) (data >> 8);
    value[7] = (byte) (data);
    return value;
  }

  /**
   * Convert a single Long object into an array of 8(?) bytes.
   * 
   * @param data
   *            The input Long
   * @return an array of bytes
   */
  public static byte[] longToByte(final Long data) {
    return longToByte(data.longValue());
  }

  /**
   * Convert a single double into an array of 8 bytes.
   * 
   * @param data
   *            The input double
   * @return an array of bytes
   */
  public static byte[] doubleToByte(final double data) {
    return null;
  }

  /**
   * Convert a single Double object into an array of 8 bytes.
   * 
   * @param data
   *            The input Double
   * @return an array of bytes
   */
  public static byte[] doubleToByte(final Double data) {
    return doubleToByte(data.doubleValue());
  }

  /**
   * Create a Number object from an array of bytes.
   * 
   * @param barray
   *            The bytes to be converted
   * @param obj
   *            Input object of the desired output class. Must be a sub-class
   *            of Number.
   * @return A Object of the type of obj.
   * @throws ConversionException
   *             if something goes wrong.
   */
  @SuppressWarnings("rawtypes")
  public static Object byteToNumber(final byte[] barray, final Object obj) throws ConversionException {
    Class theClass = obj.getClass();
    String type = theClass.getName();
    Object retobj = null;

    if (type.equals("java.lang.Integer")) {
      int[] i = Conversions.byteToInt(0, 1, barray);
      retobj = new Integer(i[0]);
    } else if (type.equals("java.lang.Byte")) {
      retobj = new Byte(barray[0]);
    } else if (type.equals("java.lang.Short")) {
      short[] f = Conversions.byteToShort(0, 1, barray);
      retobj = new Short(f[0]);
    } else if (type.equals("java.lang.Float")) {
      float[] f = Conversions.byteToFloat(0, 1, barray);
      retobj = new Float(f[0]);
    } else if (type.equals("java.lang.Long")) {
      long[] f = Conversions.byteToLong(0, 1, barray);
      retobj = new Long(f[0]);
    } else if (type.equals("java.lang.Double")) {
      double[] f = Conversions.byteToDouble(0, 1, barray);
      retobj = new Double(f[0]);
    } else {
      /* exception: unsupported type */
      ConversionException ex = new ConversionException("byteToNumber: setfield bad type: " + obj + " " + type);
      throw (ex);
    }
    return (retobj);
  }
}
