/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.validation.util.validator;

import java.lang.annotation.Annotation;
import java.util.HashMap;
import java.util.Map;

import net.sourceforge.jbizmo.commons.validation.FutureDate;
import net.sourceforge.jbizmo.commons.validation.MaxDecimalValue;
import net.sourceforge.jbizmo.commons.validation.MaxFloatValue;
import net.sourceforge.jbizmo.commons.validation.MaxIntegerValue;
import net.sourceforge.jbizmo.commons.validation.MaxLength;
import net.sourceforge.jbizmo.commons.validation.MinDecimalValue;
import net.sourceforge.jbizmo.commons.validation.MinFloatValue;
import net.sourceforge.jbizmo.commons.validation.MinIntegerValue;
import net.sourceforge.jbizmo.commons.validation.MinLength;
import net.sourceforge.jbizmo.commons.validation.NotEmpty;
import net.sourceforge.jbizmo.commons.validation.NotNull;
import net.sourceforge.jbizmo.commons.validation.PastDate;
import net.sourceforge.jbizmo.commons.validation.RegularExpression;

/**
 * <p>
 * A factory (registry) for validators of annotation-based constraints. This is a singleton for providing a single access point to
 * the registered validators. Note that this class is thread safe.
 * </p>
 * <p>
 * Copyright 2010 (C) by Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public class ValidatorFactory {
	private static ValidatorFactory instance = new ValidatorFactory();

	private final Map<Class<? extends Annotation>, ConstraintValidator> validators = new HashMap<>();

	/**
	 * Constructor
	 */
	private ValidatorFactory() {
		register(NotNull.class, new NotNullValidator());
		register(MinLength.class, new MinLengthValidator());
		register(MaxLength.class, new MaxLengthValidator());
		register(RegularExpression.class, new RegularExpressionValidator());
		register(PastDate.class, new PastDateValidator());
		register(FutureDate.class, new FutureDateValidator());
		register(MaxIntegerValue.class, new MaxIntegerValueValidator());
		register(MinIntegerValue.class, new MinIntegerValueValidator());
		register(MinFloatValue.class, new MinFloatValueValidator());
		register(MaxFloatValue.class, new MaxFloatValueValidator());
		register(MinDecimalValue.class, new MinDecimalValueValidator());
		register(MaxDecimalValue.class, new MaxDecimalValueValidator());
		register(NotEmpty.class, new NotEmptyValidator());
	}

	/**
	 * @return the singleton instance
	 */
	public static ValidatorFactory getInstance() {
		return instance;
	}

	/**
	 * Get the validator for the given constraint type or null if there is no registered validator
	 * @param constraintType
	 * @return the validator
	 */
	public ConstraintValidator getValidatorFor(Class<? extends Annotation> constraintType) {
		return validators.get(constraintType);
	}

	/**
	 * Register a validator
	 * @param constraintType
	 * @param validator
	 */
	public void register(Class<? extends Annotation> constraintType, ConstraintValidator validator) {
		validators.put(constraintType, validator);
	}

}
