/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.search.util;

import static net.sourceforge.jbizmo.commons.search.SearchService.OPERATOR_BETWEEN;
import static net.sourceforge.jbizmo.commons.search.SearchService.OPERATOR_EQUAL;
import static net.sourceforge.jbizmo.commons.search.SearchService.OPERATOR_IN;
import static net.sourceforge.jbizmo.commons.search.SearchService.OPERATOR_IS_NOT_NULL;
import static net.sourceforge.jbizmo.commons.search.SearchService.OPERATOR_IS_NULL;
import static net.sourceforge.jbizmo.commons.search.SearchService.OPERATOR_LIKE;
import static net.sourceforge.jbizmo.commons.search.SearchService.OPERATOR_NOT_IN;
import static net.sourceforge.jbizmo.commons.search.SearchService.TOKEN_DELIMITER_BETWEEN;
import static net.sourceforge.jbizmo.commons.search.SearchService.TOKEN_DELIMITER_IN;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;

import net.sourceforge.jbizmo.commons.search.dto.SearchDTO;
import net.sourceforge.jbizmo.commons.search.dto.SearchFieldDTO;
import net.sourceforge.jbizmo.commons.search.dto.SearchFieldDataTypeEnum;
import net.sourceforge.jbizmo.commons.search.dto.SearchOperatorDTO;
import net.sourceforge.jbizmo.commons.search.dto.SortDirectionEnum;
import net.sourceforge.jbizmo.commons.search.exception.GeneralSearchException;

/**
 * <p>
 * Helper class that creates JPA query statements based on given {@link SearchDTO}
 * </p>
 * <p>
 * Copyright 2010 (C) by Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public class QueryHelper
{
	public static final String WHERE = "where";
	public static final String SELECT = "select ";
	public static final String AND = " and ";
	public static final String ORDER_BY = " order by ";
	public static final String COUNT = "count(a) ";
	public static final String IN = " in (";
	public static final String BETWEEN = " between ";
	public static final String NOT_IN = " not in (";
	private static final String PARAM = "param";

	private static SearchOperatorDTO defaultOperatorEqual = new SearchOperatorDTO(0, OPERATOR_EQUAL, "equal", true, true, true, true, true);
	private static SearchOperatorDTO defaultOperatorLike = new SearchOperatorDTO(1, OPERATOR_LIKE, "like", true, false, false, true, false);

	/**
	 * Prevent instantiation
	 */
	private QueryHelper()
	{

	}

	/**
	 * @param searchDTO
	 * @return a String with a parameterized JPA query
	 * @throws GeneralSearchException if the search statement could not be created
	 */
	public static String createStatement(SearchDTO searchDTO) throws GeneralSearchException
	{
		final StringBuilder statement = new StringBuilder();
		final StringBuilder orderBy = new StringBuilder();
		boolean whereSet = false;
		String defaultOrderBy = "";

		try
		{
			if(searchDTO.getFromClause().contains(WHERE))
				whereSet = true;

			if(searchDTO.getFromClause().contains(ORDER_BY))
			{
				defaultOrderBy = searchDTO.getFromClause().substring(searchDTO.getFromClause().indexOf(ORDER_BY));

				// Extract from clause
				final String fromClause = searchDTO.getFromClause().substring(0, searchDTO.getFromClause().indexOf(ORDER_BY));
				statement.append(fromClause);
			}
			else
			{
				// Add from clause
				statement.append(searchDTO.getFromClause());
			}

			// Add where clause
			for(final SearchFieldDTO field : searchDTO.getSearchFields())
			{
				if(!processField(field))
					continue;

				statement.append(addField(field, whereSet, searchDTO.isCaseSensitive(), searchDTO.isExactFilterMatch()));

				whereSet = true;
			}

			if(searchDTO.getGroupBy() != null)
				if(!searchDTO.getGroupBy().isEmpty())
					statement.append(" " + searchDTO.getGroupBy());

			boolean isFirstOrderBy = true;

			if(!defaultOrderBy.isEmpty())
			{
				isFirstOrderBy = false;
				orderBy.append(defaultOrderBy);
			}

			// Sort fields by sort index
			Collections.sort(searchDTO.getSearchFields(), (a, b) ->
			{
				final Integer index1 = Integer.valueOf(a.getSortIndex());
				final Integer index2 = Integer.valueOf(b.getSortIndex());

				return index1.compareTo(index2);
			});

			// Add order by clause
			for(final SearchFieldDTO field : searchDTO.getSearchFields())
			{
				if(field.getSortOrder() != null && field.getSortOrder() != SortDirectionEnum.NONE)
				{
					if(!isFirstOrderBy)
						orderBy.append(",");
					else
					{
						statement.append(ORDER_BY);
						isFirstOrderBy = false;
					}

					orderBy.append(field.getColName());
					orderBy.append(" ");
					orderBy.append(field.getSortOrder().name().toLowerCase());
				}
			}

			// Check if result set should be ordered at all and add order by clause to query if necessary
			if(!isFirstOrderBy)
				statement.append(orderBy);

			return statement.toString();
		}
		catch (final Exception e)
		{
			e.printStackTrace();
			throw new GeneralSearchException(e, true);
		}
	}

	/**
	 * @param searchDTO
	 * @return a String with a parameterized JPA query for counting records
	 */
	public static String createCountStatement(SearchDTO searchDTO)
	{
		final StringBuilder statement = new StringBuilder(SELECT);
		boolean whereSet = false;

		try
		{
			statement.append(COUNT);

			// Add from clause
			statement.append(searchDTO.getFromClause());

			if(searchDTO.getFromClause().contains(WHERE))
				whereSet = true;

			// Add where clause
			for(final SearchFieldDTO field : searchDTO.getSearchFields())
			{
				if(!processField(field))
					continue;

				statement.append(addField(field, whereSet, searchDTO.isCaseSensitive(), searchDTO.isExactFilterMatch()));

				whereSet = true;
			}

			if(searchDTO.getGroupBy() != null)
				if(!searchDTO.getGroupBy().isEmpty())
					statement.append(searchDTO.getGroupBy());
		}
		catch (final Exception e)
		{
			throw new RuntimeException(e);
		}

		return statement.toString();
	}

	/**
	 * @param searchDTO
	 * @return a hash map containing the parameters according to the query statement
	 * @throws ParseException if parsing of one of the provided search field values has failed
	 */
	public static HashMap<String, Object> createParameters(SearchDTO searchDTO) throws ParseException
	{
		final HashMap<String, Object> params = new HashMap<>();

		for(final SearchFieldDTO field : searchDTO.getSearchFields())
		{
			if(!processField(field))
				continue;

			updateMap(params, field, searchDTO);
		}

		return params;
	}

	/**
	 * Determine if a search field provides appropriate data for creating a respective query fragment
	 * @param field
	 * @return true if the field should be processed
	 */
	private static boolean processField(SearchFieldDTO field)
	{
		if(field.getOperator() != null)
		{
			final SearchOperatorDTO op = field.getOperator();

			// For all operators that basically do not evaluate the filter criteria it is allowed that the respective field is null!
			// In all other cases the search field should not be processed!
			if(field.getFilterCriteria() == null && !op.getValue().equals(OPERATOR_IS_NULL) && !op.getValue().equals(OPERATOR_IS_NOT_NULL))
				return false;
		}
		else if(field.getFilterCriteria() == null || field.getFilterCriteria().isEmpty())
			return false;

		return true;
	}

	/**
	 * @param field
	 * @param exactMatch
	 * @return the in statement for given field
	 */
	private static String addInStatement(SearchFieldDTO field, boolean exactMatch)
	{
		final StringBuilder b = new StringBuilder();

		if(!field.getFilterCriteria().contains(TOKEN_DELIMITER_IN))
			return "";

		// Search for tokens
		b.append(field.getColName());

		if(field.getOperator().getValue().equals(OPERATOR_IN))
			b.append(IN);
		else
			b.append(NOT_IN);

		boolean isFirstToken = true;
		final String[] tokens = field.getFilterCriteria().split(TOKEN_DELIMITER_IN);
		int tokenIndex = 0;

		for(String token : tokens)
		{
			token = removeIllegalCharacters(token);

			if(isFirstToken)
				isFirstToken = false;
			else
				b.append(",");

			if(field.getDataType() == SearchFieldDataTypeEnum.STRING || field.getDataType() == SearchFieldDataTypeEnum.ENUM)
			{
				if(exactMatch)
				{
					b.append("'");
					b.append(token);
					b.append("'");
				}
				else
				{
					b.append("'%");
					b.append(token);
					b.append("%'");
				}
			}
			else if(field.getDataType() == SearchFieldDataTypeEnum.CHAR)
			{
				b.append("'");
				b.append(token);
				b.append("'");
			}
			else if(field.getDataType() == SearchFieldDataTypeEnum.INTEGER || field.getDataType() == SearchFieldDataTypeEnum.LONG || field.getDataType() == SearchFieldDataTypeEnum.DOUBLE
					|| field.getDataType() == SearchFieldDataTypeEnum.FLOAT || field.getDataType() == SearchFieldDataTypeEnum.BIG_DECIMAL)
				b.append(":" + PARAM + field.getColOrder() + "_" + tokenIndex++);
			else
				b.append(token);
		}

		b.append(")");

		return b.toString();
	}

	/**
	 * @param field
	 * @param exactMatch
	 * @return the between statement for given field
	 */
	private static String addBetweenStatement(SearchFieldDTO field, boolean exactMatch)
	{
		final StringBuilder buf = new StringBuilder();

		// Between filter makes no sense if no delimiter for two values exists!
		if(!field.getFilterCriteria().contains(TOKEN_DELIMITER_BETWEEN))
			return "";

		buf.append(field.getColName());
		buf.append(BETWEEN);

		if(field.getDataType() == SearchFieldDataTypeEnum.GREGORIAN_CALENDAR || field.getDataType() == SearchFieldDataTypeEnum.DATE || field.getDataType() == SearchFieldDataTypeEnum.FLOAT
				|| field.getDataType() == SearchFieldDataTypeEnum.DOUBLE || field.getDataType() == SearchFieldDataTypeEnum.INTEGER || field.getDataType() == SearchFieldDataTypeEnum.LONG
				|| field.getDataType() == SearchFieldDataTypeEnum.BIG_DECIMAL)
		{

			buf.append(":");
			buf.append(PARAM);
			buf.append(field.getColOrder() + "1");
			buf.append(AND);
			buf.append(" :");
			buf.append(PARAM);
			buf.append(field.getColOrder() + "2");

			return buf.toString();
		}

		String value1 = field.getFilterCriteria().substring(0, field.getFilterCriteria().indexOf(TOKEN_DELIMITER_BETWEEN));
		String value2 = field.getFilterCriteria().substring(field.getFilterCriteria().indexOf(TOKEN_DELIMITER_BETWEEN) + TOKEN_DELIMITER_BETWEEN.length());

		value1 = removeIllegalCharacters(value1);
		value2 = removeIllegalCharacters(value2);

		if(field.getDataType() == SearchFieldDataTypeEnum.STRING || field.getDataType() == SearchFieldDataTypeEnum.ENUM)
		{
			if(exactMatch)
			{
				buf.append("'");
				buf.append(value1);
				buf.append("'");
				buf.append(AND);
				buf.append("'");
				buf.append(value2);
				buf.append("'");
			}
			else
			{
				buf.append("'%");
				buf.append(value1);
				buf.append("%'");
				buf.append(AND);
				buf.append("'%");
				buf.append(value2);
				buf.append("%'");
			}
		}
		else if(field.getDataType() == SearchFieldDataTypeEnum.CHAR)
		{
			buf.append("'");
			buf.append(value1);
			buf.append("'");
			buf.append(AND);
			buf.append("'");
			buf.append(value2);
			buf.append("'");
		}
		else
		{
			buf.append(value1);
			buf.append(AND);
			buf.append(value2);
		}

		return buf.toString();
	}

	/**
	 * Add JPA fragment for a given field
	 * @param field
	 * @param whereSet
	 * @param caseSensitive
	 * @param exactMatch
	 * @return the created expression for that field
	 */
	private static String addField(SearchFieldDTO field, boolean whereSet, boolean caseSensitive, boolean exactMatch)
	{
		final StringBuilder buf = new StringBuilder();
		SearchOperatorDTO op = null;

		if(whereSet)
			buf.append(AND);
		else
		{
			buf.append(" ");
			buf.append(WHERE);
			buf.append(" ");
		}

		if(field.getOperator() == null)
		{
			if(field.getDataType() == SearchFieldDataTypeEnum.STRING || field.getDataType() == SearchFieldDataTypeEnum.ENUM)
				op = defaultOperatorLike;
			else
				op = defaultOperatorEqual;
		}
		else
			op = field.getOperator();

		if(op.getValue().equals(OPERATOR_IS_NULL) || op.getValue().equals(OPERATOR_IS_NOT_NULL))
		{
			// Do not add filter criteria for these operators
			buf.append(field.getColName());
			buf.append(" ");
			buf.append(op.getValue());
			buf.append(" ");
		}
		else if(op.getValue().equals(OPERATOR_BETWEEN))
			buf.append(addBetweenStatement(field, exactMatch));
		else if(op.getValue().equals(OPERATOR_IN) || op.getValue().equals(OPERATOR_NOT_IN))
			buf.append(addInStatement(field, exactMatch));
		else
		{
			if(field.getDataType() == SearchFieldDataTypeEnum.STRING || field.getDataType() == SearchFieldDataTypeEnum.ENUM)
			{
				if(caseSensitive)
				{
					if(exactMatch || op.getValue().equals(OPERATOR_EQUAL))
					{
						buf.append(field.getColName());
						buf.append(" ");
						buf.append(op.getValue());
						buf.append(" '");
						buf.append(removeIllegalCharacters(field.getFilterCriteria()));
						buf.append("'");
					}
					else
					{
						buf.append(field.getColName());
						buf.append(" ");
						buf.append(op.getValue());
						buf.append(" '%");
						buf.append(removeIllegalCharacters(field.getFilterCriteria()));
						buf.append("%'");
					}
				}
				else
				{
					if(exactMatch || op.getValue().equals(OPERATOR_EQUAL))
					{
						buf.append("lower(");
						buf.append(field.getColName());
						buf.append(") ");
						buf.append(op.getValue());
						buf.append(" '");
						buf.append(removeIllegalCharacters(field.getFilterCriteria()).toLowerCase());
						buf.append("'");
					}
					else
					{
						buf.append("lower(");
						buf.append(field.getColName());
						buf.append(") ");
						buf.append(op.getValue());
						buf.append(" '%");
						buf.append(removeIllegalCharacters(field.getFilterCriteria().toLowerCase()));
						buf.append("%'");
					}
				}
			}
			else if(field.getDataType() == SearchFieldDataTypeEnum.DATE || field.getDataType() == SearchFieldDataTypeEnum.GREGORIAN_CALENDAR || field.getDataType() == SearchFieldDataTypeEnum.FLOAT
					|| field.getDataType() == SearchFieldDataTypeEnum.DOUBLE || field.getDataType() == SearchFieldDataTypeEnum.INTEGER || field.getDataType() == SearchFieldDataTypeEnum.LONG
					|| field.getDataType() == SearchFieldDataTypeEnum.BIG_DECIMAL)
			{
				buf.append(field.getColName());
				buf.append(" ");
				buf.append(op.getValue());
				buf.append(" :");
				buf.append(PARAM);
				buf.append(field.getColOrder());
			}
			else if(field.getDataType() == SearchFieldDataTypeEnum.CHAR)
			{
				if(caseSensitive)
				{
					buf.append(field.getColName());
					buf.append(" ");
					buf.append(op.getValue());
					buf.append(" '");
					buf.append(removeIllegalCharacters(field.getFilterCriteria()));
					buf.append("'");
				}
				else
				{
					buf.append("lower(");
					buf.append(field.getColName());
					buf.append(") ");
					buf.append(op.getValue());
					buf.append(" '");
					buf.append(removeIllegalCharacters(field.getFilterCriteria()).toLowerCase());
					buf.append("'");
				}
			}
			else
			{
				buf.append(field.getColName());
				buf.append(" ");
				buf.append(op.getValue());
				buf.append(" ");
				buf.append(removeIllegalCharacters(field.getFilterCriteria()));
			}
		}

		return buf.toString();
	}

	/**
	 * Parse calendar input
	 * @param date the date value as string
	 * @param format
	 * @return the parsed {@link java.util.GregorianCalendar} value
	 * @throws ParseException if the provided date value could not be parsed
	 */
	private static GregorianCalendar parseCalendar(String date, String format) throws ParseException
	{
		final SimpleDateFormat dateFormat = new SimpleDateFormat(format);

		final Date paramDate = dateFormat.parse(date);
		final GregorianCalendar calParam = new GregorianCalendar();
		calParam.setTime(paramDate);

		return calParam;
	}

	/**
	 * Parse date
	 * @param date the date value as string
	 * @param format
	 * @return the parsed {@link java.util.Date} value
	 * @throws ParseException if the provided date value could not be parsed
	 */
	private static Date parseDate(String date, String format) throws ParseException
	{
		final SimpleDateFormat dateFormat = new SimpleDateFormat(format);

		return dateFormat.parse(date);
	}

	/**
	 * @param floatValue
	 * @param searchObj
	 * @return the parsed float value
	 * @throws ParseException if the provided float value could not be parsed
	 */
	private static float parseFloat(String floatValue, SearchDTO searchObj) throws ParseException
	{
		final DecimalFormatSymbols decimalSymbols = new DecimalFormatSymbols();
		decimalSymbols.setDecimalSeparator(searchObj.getDecimalSeparator());
		decimalSymbols.setGroupingSeparator(searchObj.getGroupingSeparator());

		final DecimalFormat decimalFormat = new DecimalFormat(searchObj.getNumberFormat(), decimalSymbols);

		return decimalFormat.parse(floatValue).floatValue();
	}

	/**
	 * @param doubleValue
	 * @param searchObj
	 * @return the parsed double value
	 * @throws ParseException if the provided double value could not be parsed
	 */
	private static double parseDouble(String doubleValue, SearchDTO searchObj) throws ParseException
	{
		final DecimalFormatSymbols decimalSymbols = new DecimalFormatSymbols();
		decimalSymbols.setDecimalSeparator(searchObj.getDecimalSeparator());
		decimalSymbols.setGroupingSeparator(searchObj.getGroupingSeparator());

		final DecimalFormat decimalFormat = new DecimalFormat(searchObj.getNumberFormat(), decimalSymbols);

		return decimalFormat.parse(doubleValue).doubleValue();
	}

	/**
	 * @param decimalValue
	 * @param searchObj
	 * @return the parsed {@link java.math.BigDecimal} value
	 * @throws ParseException if the provided big decimal value could not be parsed
	 */
	private static BigDecimal parseBigDecimal(String decimalValue, SearchDTO searchObj) throws ParseException
	{
		final DecimalFormatSymbols decimalSymbols = new DecimalFormatSymbols();
		decimalSymbols.setDecimalSeparator(searchObj.getDecimalSeparator());
		decimalSymbols.setGroupingSeparator(searchObj.getGroupingSeparator());

		final DecimalFormat decimalFormat = new DecimalFormat(searchObj.getNumberFormat(), decimalSymbols);
		decimalFormat.setParseBigDecimal(true);

		return (BigDecimal) decimalFormat.parse(decimalValue);
	}

	/**
	 * @param criterion
	 * @return a criterion without ' and " characters
	 */
	private static String removeIllegalCharacters(String criterion)
	{
		return criterion.replace("\'", "").replace("\"", "");
	}

	/**
	 * Private method that updates the parameter map
	 * @param params
	 * @param field
	 * @param searchDTO
	 * @throws ParseException if parsing of a floating point or date value fails
	 * @throws NumberFormatException if conversion of an integer value fails
	 */
	private static void updateMap(HashMap<String, Object> params, SearchFieldDTO field, SearchDTO searchDTO) throws ParseException
	{
		SearchOperatorDTO op = null;
		final SearchFieldDataTypeEnum dataType = field.getDataType();

		if(field.getOperator() == null)
		{
			if(dataType == SearchFieldDataTypeEnum.STRING || dataType == SearchFieldDataTypeEnum.ENUM)
				op = defaultOperatorLike;
			else
				op = defaultOperatorEqual;
		}
		else
			op = field.getOperator();

		if((dataType == SearchFieldDataTypeEnum.GREGORIAN_CALENDAR || dataType == SearchFieldDataTypeEnum.DATE || dataType == SearchFieldDataTypeEnum.DOUBLE || dataType == SearchFieldDataTypeEnum.FLOAT
				|| dataType == SearchFieldDataTypeEnum.INTEGER || dataType == SearchFieldDataTypeEnum.LONG || dataType == SearchFieldDataTypeEnum.BIG_DECIMAL)
				&& !(op.getValue().equals(OPERATOR_IS_NULL) || op.getValue().equals(OPERATOR_IS_NOT_NULL)))
		{
			String format;

			if(dataType == SearchFieldDataTypeEnum.GREGORIAN_CALENDAR || dataType == SearchFieldDataTypeEnum.DATE)
			{
				if(field.isDateTimeFormat())
					format = searchDTO.getDateTimeFormat();
				else
					format = searchDTO.getDateFormat();
			}
			else
				format = searchDTO.getNumberFormat();

			// Add special behavior for date and decimal fields as they need to be parameterized!
			if(field.getFilterCriteria().contains(TOKEN_DELIMITER_BETWEEN) && op.getValue().equals(OPERATOR_BETWEEN))
			{
				final String value1 = field.getFilterCriteria().substring(0, field.getFilterCriteria().indexOf(TOKEN_DELIMITER_BETWEEN));
				final String value2 = field.getFilterCriteria().substring(field.getFilterCriteria().lastIndexOf(TOKEN_DELIMITER_BETWEEN) + TOKEN_DELIMITER_BETWEEN.length(),
						field.getFilterCriteria().length());

				if(dataType == SearchFieldDataTypeEnum.GREGORIAN_CALENDAR)
				{
					params.put(PARAM + field.getColOrder() + "1", parseCalendar(value1, format));
					params.put(PARAM + field.getColOrder() + "2", parseCalendar(value2, format));
				}
				else if(dataType == SearchFieldDataTypeEnum.DATE)
				{
					params.put(PARAM + field.getColOrder() + "1", parseDate(value1, format));
					params.put(PARAM + field.getColOrder() + "2", parseDate(value2, format));
				}
				else if(dataType == SearchFieldDataTypeEnum.FLOAT)
				{
					params.put(PARAM + field.getColOrder() + "1", parseFloat(value1, searchDTO));
					params.put(PARAM + field.getColOrder() + "2", parseFloat(value2, searchDTO));
				}
				else if(dataType == SearchFieldDataTypeEnum.DOUBLE)
				{
					params.put(PARAM + field.getColOrder() + "1", parseDouble(value1, searchDTO));
					params.put(PARAM + field.getColOrder() + "2", parseDouble(value2, searchDTO));
				}
				else if(dataType == SearchFieldDataTypeEnum.BIG_DECIMAL)
				{
					params.put(PARAM + field.getColOrder() + "1", parseBigDecimal(value1, searchDTO));
					params.put(PARAM + field.getColOrder() + "2", parseBigDecimal(value2, searchDTO));
				}
				else if(dataType == SearchFieldDataTypeEnum.INTEGER)
				{
					params.put(PARAM + field.getColOrder() + "1", Integer.parseInt(value1));
					params.put(PARAM + field.getColOrder() + "2", Integer.parseInt(value2));
				}
				else if(dataType == SearchFieldDataTypeEnum.LONG)
				{
					params.put(PARAM + field.getColOrder() + "1", Long.parseLong(value1));
					params.put(PARAM + field.getColOrder() + "2", Long.parseLong(value2));
				}
			}
			else if(field.getFilterCriteria().contains(TOKEN_DELIMITER_IN) && (op.getValue().equals(OPERATOR_IN) || op.getValue().equals(OPERATOR_NOT_IN)))
			{
				final String[] tokens = field.getFilterCriteria().split(TOKEN_DELIMITER_IN);
				int tokenIndex = 0;

				for(final String token : tokens)
					if(dataType == SearchFieldDataTypeEnum.FLOAT)
						params.put(PARAM + field.getColOrder() + "_" + tokenIndex++, parseFloat(token, searchDTO));
					else if(dataType == SearchFieldDataTypeEnum.DOUBLE)
						params.put(PARAM + field.getColOrder() + "_" + tokenIndex++, parseDouble(token, searchDTO));
					else if(dataType == SearchFieldDataTypeEnum.BIG_DECIMAL)
						params.put(PARAM + field.getColOrder() + "_" + tokenIndex++, parseBigDecimal(token, searchDTO));
					else if(dataType == SearchFieldDataTypeEnum.INTEGER)
						params.put(PARAM + field.getColOrder() + "_" + tokenIndex++, Integer.parseInt(token));
					else if(dataType == SearchFieldDataTypeEnum.LONG)
						params.put(PARAM + field.getColOrder() + "_" + tokenIndex++, Long.parseLong(token));
			}
			else
			{
				if(dataType == SearchFieldDataTypeEnum.GREGORIAN_CALENDAR)
					params.put(PARAM + field.getColOrder(), parseCalendar(field.getFilterCriteria(), format));
				else if(dataType == SearchFieldDataTypeEnum.DATE)
					params.put(PARAM + field.getColOrder(), parseDate(field.getFilterCriteria(), format));
				else if(dataType == SearchFieldDataTypeEnum.FLOAT)
					params.put(PARAM + field.getColOrder(), parseFloat(field.getFilterCriteria(), searchDTO));
				else if(dataType == SearchFieldDataTypeEnum.DOUBLE)
					params.put(PARAM + field.getColOrder(), parseDouble(field.getFilterCriteria(), searchDTO));
				else if(dataType == SearchFieldDataTypeEnum.BIG_DECIMAL)
					params.put(PARAM + field.getColOrder(), parseBigDecimal(field.getFilterCriteria(), searchDTO));
				else if(dataType == SearchFieldDataTypeEnum.INTEGER)
					params.put(PARAM + field.getColOrder(), Integer.parseInt(field.getFilterCriteria()));
				else if(dataType == SearchFieldDataTypeEnum.LONG)
					params.put(PARAM + field.getColOrder(), Long.parseLong(field.getFilterCriteria()));
			}
		}
	}

}
