/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.property;

import java.io.IOException;
import java.io.InputStream;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;

/**
 * <p>
 * Simple utility that provides methods to access properties. Note that some properties have to be provided in a predefined format within the properties file: java.util.Date: "dd.MM.yyyy HH:mm:ss"
 * double: "0.00" float: "0.00" Furthermore it is important that the application that uses this service has to provide a property file at a predefined location within the library!
 * </p>
 * <p>
 * Copyright 2010 (C) by Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public class PropertyService
{
	public static final String PROP_LOGGING_LEVEL = "logging_level";
	public static final String PROP_EXCHANGE_FOLDER = "exchange_folder";
	public static final String PROP_REPOSITORY_FOLDER = "document_repository_folder";

	private static boolean propertiesLoaded = false;
	private static Properties properties = null;
	private static final String PROPERTY_LOCATION = "config/application.properties";
	private static final String DATE_FORMAT = "dd.MM.yyyy HH:mm:ss";
	private static final String DECIMAL_FORMAT = "0.00";
	private static SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT);
	private static DecimalFormat decimalFormat = new DecimalFormat(DECIMAL_FORMAT);

	/**
	 * Disable instantiation
	 */
	private PropertyService()
	{

	}

	/**
	 * Get property
	 * @param name the property name
	 * @return the property value
	 * @throws IllegalPropertyValueException it the property doesn't exist
	 */
	private static String getProperty(String name) throws IllegalPropertyValueException
	{
		if(!propertiesLoaded)
			loadProperties();

		if(properties.getProperty(name) == null)
			throw new IllegalPropertyValueException("Property " + name + " does not exist!");

		return properties.getProperty(name);
	}

	/**
	 * Get property
	 * @param name the property name
	 * @return the property value
	 * @throws IllegalPropertyValueException if the property doesn't exist
	 */
	public static String getStringProperty(String name) throws IllegalPropertyValueException
	{
		return getProperty(name);
	}

	/**
	 * Get property
	 * @param name the property name
	 * @return the property value
	 * @throws IllegalPropertyValueException if the property value doesn't represent an integer
	 */
	public static int getIntProperty(String name) throws IllegalPropertyValueException
	{
		final String propertyValue = getProperty(name);

		try
		{
			return Integer.parseInt(propertyValue);
		}
		catch (final NumberFormatException e)
		{
			throw new IllegalPropertyValueException("Property " + name + " does not represent a valid integer value!");
		}
	}

	/**
	 * Get property
	 * @param name the property name
	 * @return the property value
	 * @throws IllegalPropertyValueException if the property value doesn't represent a long
	 */
	public static long getLongProperty(String name) throws IllegalPropertyValueException
	{
		final String propertyValue = getProperty(name);

		try
		{
			return Long.parseLong(propertyValue);
		}
		catch (final NumberFormatException e)
		{
			throw new IllegalPropertyValueException("Property " + name + " does not represent a valid long value!");
		}
	}

	/**
	 * Get property
	 * @param name the property name
	 * @return the property value
	 * @throws IllegalPropertyValueException if the property value doesn't represent a double
	 */
	public static double getDoubleProperty(String name) throws IllegalPropertyValueException
	{
		final String propertyValue = getProperty(name);

		try
		{
			final Number n = decimalFormat.parse(propertyValue);
			return n.doubleValue();
		}
		catch (final ParseException e)
		{
			throw new IllegalPropertyValueException("Property " + name + " does not represent a valid double value!");
		}
	}

	/**
	 * Get property
	 * @param name the property name
	 * @return the property value
	 * @throws IllegalPropertyValueException if the property value doesn't represent a float
	 */
	public static float getFloatProperty(String name) throws IllegalPropertyValueException
	{
		final String propertyValue = getProperty(name);

		try
		{
			final Number n = decimalFormat.parse(propertyValue);
			return n.floatValue();
		}
		catch (final ParseException e)
		{
			throw new IllegalPropertyValueException("Property " + name + " does not represent a valid float value!");
		}
	}

	/**
	 * Get property
	 * @param name the property name
	 * @return the property value
	 * @throws IllegalPropertyValueException if the property value doesn't represent a character
	 */
	public static char getCharProperty(String name) throws IllegalPropertyValueException
	{
		final String propertyValue = getProperty(name);

		if(propertyValue.isEmpty() || propertyValue.length() > 1)
			throw new IllegalPropertyValueException("Property " + name + " does not represent a valid character value!");

		return propertyValue.charAt(0);
	}

	/**
	 * Get property
	 * @param name the property name
	 * @return the property value
	 * @throws IllegalPropertyValueException if the property value doesn't represent a date
	 */
	public static synchronized Date getDateProperty(String name) throws IllegalPropertyValueException
	{
		final String propertyValue = getProperty(name);

		try
		{
			return dateFormat.parse(propertyValue);
		}
		catch (final ParseException e)
		{
			throw new IllegalPropertyValueException("Property " + name + " does not represent a valid java.util.Date value!");
		}
	}

	/**
	 * Get property
	 * @param name the property name
	 * @return the property value
	 * @throws IllegalPropertyValueException if the property value doesn't represent a boolean
	 */
	public static boolean getBooleanProperty(String name) throws IllegalPropertyValueException
	{
		final String propertyValue = getProperty(name);

		return Boolean.parseBoolean(propertyValue);
	}

	/**
	 * Load properties from library file
	 */
	private static synchronized void loadProperties()
	{
		final InputStream in = PropertyService.class.getClassLoader().getResourceAsStream(PROPERTY_LOCATION);

		if(in == null)
			return;

		properties = new Properties();

		try
		{
			properties.load(in);
		}
		catch (final IOException e)
		{
			e.printStackTrace();
			return;
		}

		propertiesLoaded = true;
	}

}
