/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.stream;

import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;

import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;

import net.sourceforge.jbizmo.commons.crypto.CipherFactory;

/**
 * <p>
 * Utility class to provide common operations upon streams that are used for communication between client and server.
 * </p>
 * <p>
 * Copyright 2010 (C) Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public class StreamWorker
{
	public static final int DEFAULT_BUFFER_SIZE = 1024;
	private final int bufferSize;

	/**
	 * Default constructor
	 */
	public StreamWorker()
	{
		this(DEFAULT_BUFFER_SIZE);
	}

	/**
	 * Constructor to define size of internal buffer
	 * @param bufferSize
	 */
	public StreamWorker(int bufferSize)
	{
		this.bufferSize = bufferSize;
	}

	/**
	 * Write data from input stream to output stream
	 * @param in
	 * @param out
	 * @param encryptOutput
	 * @param decryptInput
	 * @throws Exception
	 */
	public void writeToOutput(InputStream in, OutputStream out, boolean encryptOutput, boolean decryptInput) throws Exception
	{
		final byte[] buf = new byte[bufferSize <= 0 ? DEFAULT_BUFFER_SIZE : bufferSize];
		int numRead = 0;

		try
		{
			if(encryptOutput)
				out = new CipherOutputStream(out, CipherFactory.getEncryptionCipher());

			if(decryptInput)
				in = new CipherInputStream(in, CipherFactory.getDecryptionCipher());

			while((numRead = in.read(buf)) >= 0)
				out.write(buf, 0, numRead);
		}
		finally
		{
			out.close();
		}
	}

	/**
	 * Read serialized object from input stream
	 * @param in
	 * @param decrypt
	 * @return the object
	 * @throws Exception
	 */
	public Object readObjectFromStream(InputStream in, boolean decrypt) throws Exception
	{
		ObjectInputStream ois = null;

		try
		{
			if(decrypt)
				ois = new ObjectInputStream(new CipherInputStream(in, CipherFactory.getDecryptionCipher()));
			else
				ois = new ObjectInputStream(in);

			return ois.readObject();
		}
		finally
		{
			if(ois != null)
				ois.close();
		}
	}

	/**
	 * Write object to stream
	 * @param object
	 * @param out
	 * @param encrypt
	 * @throws Exception
	 */
	public void writeObjectToStream(Serializable object, OutputStream out, boolean encrypt) throws Exception
	{
		ObjectOutputStream oos = null;

		try
		{
			if(encrypt)
				oos = new ObjectOutputStream(new CipherOutputStream(out, CipherFactory.getEncryptionCipher()));
			else
				oos = new ObjectOutputStream(out);

			oos.writeObject(object);
		}
		finally
		{
			if(oos != null)
				oos.close();
		}
	}
}
