/*
 * This file is part of JBizMo, a set of tools, libraries and plug-ins
 * for modeling and creating Java-based enterprise applications.
 * For more information visit:
 *
 * http://sourceforge.net/projects/jbizmo/
 *
 * This software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package net.sourceforge.jbizmo.commons.reflect;

import java.lang.reflect.Method;
import java.util.Hashtable;
import java.util.Vector;

/**
 * <p>
 * Utility class for finding method by method name and parameters
 * </p>
 * <p>
 * Copyright 2010 (C) by Martin Ganserer
 * </p>
 * @author Martin Ganserer
 * @version 1.0.0
 */
public class MethodFinder
{
	/**
	 * Private constructor in order to avoid instantiation
	 */
	private MethodFinder()
	{

	}

	/**
	 * Find method
	 * @param cl the class
	 * @param methodName the name of the method
	 * @param params the parameters
	 * @return the method
	 * @throws NoSuchMethodException
	 */
	public static Method findMethod(Class<?> cl, String methodName, Object[] params) throws NoSuchMethodException
	{
		final Hashtable<String, Vector<Method>> methodHash = new Hashtable<>();
		final Hashtable<Method, Object[]> paramHash = new Hashtable<>();

		// Get all class methods
		final Method[] methods = cl.getMethods();

		// Build method hash map
		for(int i = 0; i < methods.length; i++)
		{
			final String mName = methods[i].getName();

			Vector<Method> v = methodHash.get(mName);

			if(v == null)
			{
				v = new Vector<>();
				methodHash.put(mName, v);
			}

			v.addElement(methods[i]);
			final Class<?>[] paramTypes = methods[i].getParameterTypes();
			paramHash.put(methods[i], paramTypes);
		}

		final Vector<Method> v = methodHash.get(methodName);

		if(v == null)
			throw new NoSuchMethodException(methodName);

		// Find method
		for(int i = 0; i < v.size(); i++)
		{
			final Method m = v.elementAt(i);
			boolean ok = true;
			final Class<?>[] paramTypes = (Class[]) paramHash.get(m);

			if(params == null)
				params = new Object[0];

			if(paramTypes.length != params.length)
				continue;

			for(int j = 0; j < paramTypes.length; j++)
			{
				if(params[j] == null)
					continue;

				final Class<?> c = params[j].getClass();
				final Class<?> paramClass = paramTypes[j];

				if(paramClass.isPrimitive())
				{
					if(paramClass.equals(Integer.TYPE))
					{
						if(c.equals(Integer.class))
							continue;

						ok = false;
						break;
					}
					else if(paramClass.equals(Double.TYPE))
					{
						if(c.equals(Double.class))
							continue;

						ok = false;
						break;
					}
					else if(paramClass.equals(Boolean.TYPE))
					{
						if(c.equals(Boolean.class))
							continue;

						ok = false;
						break;
					}
					else if(paramClass.equals(Long.TYPE))
					{
						if(c.equals(Long.class))
							continue;

						ok = false;
						break;
					}
					else if(paramClass.equals(Character.TYPE))
					{
						if(c.equals(Character.class))
							continue;

						ok = false;
						break;
					}
					else if(paramClass.equals(Byte.TYPE))
					{
						if(c.equals(Byte.class))
							continue;

						ok = false;
						break;
					}
					else if(paramClass.equals(Float.TYPE))
					{
						if(c.equals(Float.class))
							continue;

						ok = false;
						break;
					}
					else if(paramClass.equals(Short.TYPE))
					{
						if(c.equals(Short.class))
							continue;

						ok = false;
						break;
					}

					break;
				}

				if(!paramClass.isAssignableFrom(c))
				{
					ok = false;
					break;
				}
			}

			if(ok)
				return m;
		}

		throw new NoSuchMethodException(methodName);
	}

}
