/* ==================================================================
 * ModbusMessage.java - 25/11/2022 3:10:59 pm
 *
 * Copyright 2022 SolarNetwork.net Dev Team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 * ==================================================================
 */

package net.solarnetwork.io.modbus;

/**
 * API for a Modbus message.
 *
 * @author matt
 * @version 1.0
 */
public interface ModbusMessage {

	/**
	 * Get the device unit ID.
	 *
	 * <p>
	 * Modbus allows only values from 0-255 for the unit ID.
	 * </p>
	 *
	 * @return the unit ID, as an integer
	 */
	int getUnitId();

	/**
	 * Get the Modbus function code.
	 *
	 * @return the function code, never {@literal null}
	 */
	ModbusFunction getFunction();

	/**
	 * Get the Modbus error code.
	 *
	 * @return the error code, or {@literal null} if not an error
	 */
	ModbusError getError();

	/**
	 * Test if this message is an exception (error).
	 *
	 * @return {@literal true} if {@link #getError()} it not {@literal null}
	 */
	default boolean isException() {
		return (getError() != null);
	}

	/**
	 * Unwrap this message as a specific message type, if possible.
	 *
	 * <p>
	 * This method should be used instead of relying on a direct
	 * {@code instanceof} operator, because the actual Modbus message may be
	 * encapsulated in some way. For example, instead of trying this:
	 * </p>
	 *
	 * <pre>
	 * <code>
	 * // WRONG WAY: DO NOT TRY THIS
	 * ModbusMessage msg = getMessageFromSomewhere();
	 * if ( msg instanceof RegistersModbusMessage ) {
	 *   RegistersModbusMessage r = (RegistersModbusMessage)msg;
	 *   // do something with registers...
	 * }
	 * </code>
	 * </pre>
	 *
	 * <p>
	 * try this instead:
	 * </p>
	 *
	 * <pre>
	 * <code>
	 * ModbusMessage msg = getMessageFromSomewhere();
	 * RegistersModbusMessage r = msg.unwrap(RegistersModbusMessage.class);
	 * if ( r != null ) {
	 *   // do something with registers...
	 * }
	 * </code>
	 * </pre>
	 *
	 * @param <T>
	 *        the message type to unwrap
	 * @param msgType
	 *        the class to unwrap as
	 * @return the message as the given type, or {@literal null} if this message
	 *         is not compatible with {@code msgType}
	 */
	<T extends ModbusMessage> T unwrap(Class<T> msgType);

	/**
	 * Compare the "sameness" of this message to another.
	 *
	 * @param obj
	 *        the message to compare to this message
	 * @return if the class of {@code other} is the same as this class, and the
	 *         properties are the same
	 */
	boolean isSameAs(ModbusMessage obj);

	/**
	 * Validate this message in some way.
	 *
	 * <p>
	 * This method might validate a checksum or the overall structure of the
	 * message for correctness, throwing an exception if any validation fails.
	 * </p>
	 *
	 * @return this message
	 * @throws ModbusValidationException
	 *         if any validation failure occurs
	 */
	default ModbusMessage validate() throws ModbusValidationException {
		// implementing classes can override
		return this;
	}

	@Override
	int hashCode();

	@Override
	boolean equals(Object obj);

}
