/*
 * Decompiled with CFR 0.152.
 */
package net.solarnetwork.central.support;

import java.io.IOException;
import java.nio.file.FileStore;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Duration;
import java.time.Instant;
import java.util.LinkedHashMap;
import net.solarnetwork.central.scheduler.JobSupport;
import net.solarnetwork.service.PingTest;
import net.solarnetwork.service.PingTestResult;
import net.solarnetwork.util.ObjectUtils;
import org.springframework.util.unit.DataSize;

public class DirectoryCleanerJob
extends JobSupport
implements PingTest {
    public static final Duration DEFAULT_MINIMUM_AGE = Duration.ofMinutes(60L);
    public static final DataSize DEFAULT_FREE_SPACE_WARNING_SIZE = DataSize.ofMegabytes((long)100L);
    private final Path directory;
    private Duration minimumAge;
    private DataSize freeSpaceWarningSize;
    private long fileDeleteCount;

    public DirectoryCleanerJob(Path directory) {
        this.directory = (Path)ObjectUtils.requireNonNullArgument((Object)directory, (String)"directory");
        this.minimumAge = DEFAULT_MINIMUM_AGE;
        this.freeSpaceWarningSize = DEFAULT_FREE_SPACE_WARNING_SIZE;
        this.fileDeleteCount = 0L;
    }

    @Override
    public synchronized void run() {
        final Instant expireDate = Instant.now().minus(this.minimumAge);
        try {
            Files.walkFileTree(this.directory, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (Files.getLastModifiedTime(file, new LinkOption[0]).toInstant().isBefore(expireDate) && Files.deleteIfExists(file)) {
                        DirectoryCleanerJob.this.log.info("Deleted file: [{}]", (Object)file);
                        ++DirectoryCleanerJob.this.fileDeleteCount;
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    DirectoryCleanerJob.this.log.warn("IO error inspecting file [{}] for deletion: {}", (Object)file, (Object)exc.toString());
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            this.log.error("IO error cleaning files older than {} from [{}]: {}", new Object[]{this.minimumAge, this.directory, e.toString()});
        }
    }

    public String getPingTestId() {
        return this.getClass().getName() + "-" + String.valueOf(this.directory);
    }

    public String getPingTestName() {
        return "Directory Cleaner";
    }

    public long getPingTestMaximumExecutionMilliseconds() {
        return 2000L;
    }

    public PingTest.Result performPingTest() throws Exception {
        DataSize size = this.getFreeSpaceWarningSize();
        LinkedHashMap<String, Object> props = new LinkedHashMap<String, Object>(4);
        props.put("FilesDeleted", this.fileDeleteCount);
        if (size == null || size.toBytes() < 1L) {
            return new PingTestResult(true, "Disk space monitoring disabled.", props);
        }
        FileStore fs = Files.getFileStore(this.directory);
        long freeSpace = fs.getUsableSpace();
        props.put("FreeSpace", String.format("%d MB", DataSize.ofBytes((long)freeSpace).toMegabytes()));
        if (freeSpace < size.toBytes()) {
            return new PingTestResult(false, String.format("Disk space below %s.", size), props);
        }
        return new PingTestResult(true, "Disk space OK.", props);
    }

    public void setMinutesOlder(int minutes) {
        this.setMinimumAge(Duration.ofMinutes(minutes));
    }

    public Duration getMinimumAge() {
        return this.minimumAge;
    }

    public void setMinimumAge(Duration minimumAge) {
        this.minimumAge = minimumAge != null ? minimumAge : DEFAULT_MINIMUM_AGE;
    }

    public DataSize getFreeSpaceWarningSize() {
        return this.freeSpaceWarningSize;
    }

    public void setFreeSpaceWarningSize(DataSize freeSpaceWarningSize) {
        this.freeSpaceWarningSize = freeSpaceWarningSize;
    }
}

