package net.sodacan.core.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <p>This BTree can exist on disk. Changes made in memory don't immediately
 * get sent to disk. Rather, commit is done on commit. Only changed
 * nodes are written.</p>
 * <p>A "new" node has two meanings. New-to-memory and completely-new.
 * A node that's new to memory will have a reference pointer while a completely
 * new node will not at the time it is created.</p>
 * <p>In order to avoid bringing the whole tree into memory when we access the root node
 * nodes children are realized lazily.</p>
 */
public class BBBTree {
	public static final int INITIAL_NODE_ALLOCATION = 500;
	private BBAlloc a;
	private Set<Node> changedNodes = new HashSet<>();
	private Node root = null;
	private Map<Integer,Node> nodesByRef = new HashMap<>();
	/*
	 * Keep this real simple to start
	 */
	public BBBTree(BBAlloc a) {
		this.a = a;
	}

	public void store(Integer ref, String serialized) {
		a.store(ref, serialized);
	}

	public Integer store(String serialized) {
		return a.store(serialized);
	}
	
	public String getNodeString(Integer ref) {
		return a.fetch(ref);
	}

	/**
	 * Save all the updated and new nodes
	 */
	public void commit() {
		// First, everyone needs a ref, this does it recursively
		for (Node node : changedNodes) {
			node.assignReference();
		}
		// Now we can safely save everything that's changed
		for (Node node : changedNodes) {
			node.saveNode();
		}
		changedNodes.clear();
	}

	public BBAlloc getAlloc() {
		return a;
	}

	public Set<Node> getChangedNodes() {
		return changedNodes;
	}
	
	public Node getRoot() {
		if (root!=null) return root;
		// Two reasons root could be null: a new tree
		// Or an existing tree that has not been realized yet.
		if (a.count()==0) {
			// Make this a long string so there's room to replace it with just a serialized number
			// We don't actually save the root 'til we're asked to.
			a.store("A place to store root ref");
			root = new Node(this);
		} else {
			String rootString = a.fetch(0);
			Integer ref = Integer.valueOf(rootString);
			root = new Node(this, ref);
		}
		return root;
	}

	public void setRoot(Node root) {
		this.root = root;
	}

	public static class Node {
		boolean realized;
		Integer ref;
		BBBTree tree;
		String name;
		Node parent;
		List<Node> children;
	
		/*
		 * Create a completely new node
		 */
		public Node(BBBTree tree) {
			this.realized = true;
			this.tree = tree;
			this.name = null;
			this.ref = null;
			this.children = new LinkedList<>();
			this.tree.changedNodes.add(this);
		}

		/*
		 * Create a completely new node
		 */
		public Node(BBBTree tree, Node parent) {
			this(tree);
			this.parent = parent;
			parent.addChild(this);
		}
		
		/**
		 * Create a skeleton node which won't be realized until it's first needed
		 * @param tree
		 * @param ref
		 */
		public Node(BBBTree tree, Integer ref) {
			this.realized = false;
			this.tree = tree;
			this.ref = ref;
			tree.nodesByRef.put(ref, this);
			this.children = new LinkedList<>();
		}

		public void setName(String name) {
			realize();
			this.name = name;
			this.tree.changedNodes.add(this);
		}
		
		public void addChild(Node child) {
			realize();
			this.children.add(child);
			this.tree.changedNodes.add(this);
		}

		/**
		 * Populate a node from a serialized form from disk
		 * Can be called repeatedly, does nothing if already done.
		 * Our parent and children may or may not already exist in memory
		 * 
		 */
		public void realize()  {
			if (realized) return;
			if (ref==null) {
				throw new RuntimeException("Missing ref for realization");
			}
			String nodeString = tree.getNodeString(ref);
			String f[] = nodeString.split("\\|");
			name = f[0];
			if (f[1] !=null && !f[1].isEmpty()) {
				int parentRef = Integer.parseInt(f[1]);
				parent = tree.nodesByRef.get(parentRef);
				if (parent==null) {
					parent = new Node(tree, parentRef);
				}
			}
			for (int x=2;x<f.length;x++) {
				int childRef = Integer.parseInt(f[x]);
				Node child = tree.nodesByRef.get(childRef);
				if (child==null) {
					child = new Node(tree, childRef);
				}
				this.children.add(child);
			}
			realized = true;
		}

		/**
		 * Make sure this node has a ref
		 */
		public void assignReference() {
			if (ref==null) {
				ref = tree.getAlloc().alloc(0);
			}
			tree.nodesByRef.put(ref, this);
		}
		
		/**
		 * By the time we're called, every node should have a ref assigned
		 */
		public void saveNode() {
			realize();
			StringBuffer sb = new StringBuffer();
			sb.append(name);
			sb.append('|');
			if (parent!=null) {
				sb.append(parent.ref);
			}
			for (Node child : children) {
				sb.append('|');
				sb.append(child.ref.toString());
			}
			tree.store(ref, sb.toString());
			// Special handling if this is the root node
			if (parent==null) {
				// Need to remember where the root node is.
				tree.store(0, Integer.toString(ref));
			}
		}
		
		public Integer getRef() {
			return ref;
		}

		public String getName() {
			realize();
			return name;
		}

		public Node getParent() {
			realize();
			return parent;
		}

		public List<Node> getChildren() {
			realize();
			return children;
		}

		@Override
		public String toString() {
			// TODO Auto-generated method stub
			return "(" + ref + ") "+ name;
		}
		
	}
}
