package net.sodacan.core.util;

import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;

public class BBAlloc implements Closeable {
	public static final int HEADER_SIZE = 1024;
	public static final int INT_SIZE = 4;
	public static final int MAGIC_NUMBER = 3275;
	int version = 1;
	int magicNumber;
	int refMax;
	int spaceMax;
	int refBase;
	int freeSpaceBase;
	int nextRef;
	int nextFreeSpace;
	int wastedSpace = 0;
	ByteBuffer space;

	/**
	 * Use an existing bytebuffer
	 * @param space
	 */
	public BBAlloc(ByteBuffer space) {
		this.space = space;
		restoreHeader();
	}

	public BBAlloc(ByteBuffer space, int refMax, int spaceMax) {
		this.space = space;
		// Lets see if this is an existing file
		if (MAGIC_NUMBER==space.getInt()) {
			// Existing file
			restoreHeader();
			// But we can still update spaceMax, not refMax without a reorg
			this.spaceMax = spaceMax;
		} else {
			// New file
			this.magicNumber = MAGIC_NUMBER;
			this.spaceMax = spaceMax;
			this.refBase = HEADER_SIZE;
			this.nextRef = 0;
			this.refMax = refMax;
			this.freeSpaceBase = refBase + (refMax*INT_SIZE);
			this.nextFreeSpace = freeSpaceBase;
			saveHeader();
		}
	}
	
	/**
	 * Copy constructor, compacting space as we go
	 * @param src
	 * @param space
	 */
	public BBAlloc(BBAlloc src, ByteBuffer space, int refMax, int spaceMax) {
		this.version = 1;
		this.spaceMax = spaceMax;
		this.space = space;
		this.refBase = HEADER_SIZE;
		this.nextRef = 0;
		this.refMax = refMax;
		this.freeSpaceBase = refBase + (refMax*INT_SIZE);
		this.nextFreeSpace = freeSpaceBase;
		saveHeader();
		// Copy everything over-preserve ref position
		for(int x=0;x<src.count();x++) {
			store(src.fetch(x));
		}		
	}
	/**
	 * Must remain sync with saveHeader.
	 * Also, version must be first so we can decide if we need
	 * to upgrade something
	 */
	private void restoreHeader() {
		space.rewind();
		this.magicNumber = space.getInt();
		this.version = space.getInt();
		this.spaceMax = space.getInt();
		this.refMax = space.getInt();
		this.refBase = space.getInt();
		this.freeSpaceBase = space.getInt();
		this.nextRef = space.getInt();
		this.nextFreeSpace = space.getInt();
		this.wastedSpace = space.getInt();
	}

	/**
	 * Must remain in sync with restoreHeader.
	 * Also, version must be first so we can decide if we need
	 * to upgrade something
	 */
	public void saveHeader() {
		space.rewind();
		space.putInt(magicNumber);
		space.putInt(version);
		space.putInt(spaceMax);
		space.putInt(refMax);
		space.putInt(refBase);
		space.putInt(freeSpaceBase);
		space.putInt(nextRef);
		space.putInt(nextFreeSpace);
		space.putInt(wastedSpace);
	}
	
	public void setPointerAddress(int ref, int offset) {
		int p = refBase + (ref*INT_SIZE);
		space.putInt(p, offset);	
	}
	
	/**
	 * Allocate space and return a reference to it.
	 * We always allocate INT_SIZE extra bytes to hold the length of the allocation.
	 * @param size of the space requested or zero to just reserve a reference
	 * @return
	 */
	public int alloc(int size) {
		// Do we have room for another pointer
		if (nextRef>=refMax) {
			throw new RuntimeException("No room for another reference to be added");
		}
		if (nextFreeSpace+size>=spaceMax) {
			throw new RuntimeException("No room for allocation size=" + size);
		}
		if (size<0) {
			throw new RuntimeException("Allocation size must be zero or greater");
		}
		// The next pointer will go here
		int p = refBase + (nextRef*INT_SIZE);
		// If size is zero, don't allocate storage yet
		if (size==0) {
			space.putInt(p, 0);
		} else {
			int base = nextFreeSpace;
			// Keep track of the reference in the pointer array
			space.putInt(p, base);
			// Store the size of the allocation at the allocation site
			space.putInt(base, size);
			nextFreeSpace+=size+INT_SIZE; // The size requested plus an additional int
		}
		int ref = nextRef++;
		return ref;
	}
	
	/**
	 * Return an address in space, the address is the actual data, 
	 * not including the length
	 * @param ref
	 * @return
	 */
	public int getAddress(int ref) {
		if (ref < 0 || ref >= this.nextRef) {
			throw new RuntimeException("reference is out of range");
		}
		int p = refBase + (ref*INT_SIZE);
		return space.getInt(p) + INT_SIZE; // Move past the size field
	}
	
	/**
	 * Return the size of a particular allocation which could be zero if there is none
	 * yet.
	 * @param ref
	 * @return
	 */
	public int getSize(int ref) {
		if (ref < 0 || ref >= this.nextRef) {
			throw new RuntimeException("reference is out of range");
		}
		int p = refBase + (ref*INT_SIZE);
		int base = space.getInt(p);
		if (base==0) {
			return 0;
		} else {
			return space.getInt(base);
		}
	}
	
	/**
	 * Store a string and return a reference to it
	 * @param src
	 * @return
	 */
	public int store(String src) {
		byte[] srcBytes = src.getBytes();
		int size = srcBytes.length;
		int ref = alloc(size);
		int dest = getAddress(ref);
		space.put(dest, srcBytes);
		return ref;
	}

	/**
	 * Store a string at the supplied reference
	 * @param src
	 * @param ref
	 */
	public void store(Integer ref, String src) {
		byte[] srcBytes = src.getBytes();
		if (getSize(ref)==0) {
			// No allocation yet, so we do it now
			setPointerAddress(ref,nextFreeSpace);
			space.putInt(nextFreeSpace, srcBytes.length);	// size of the allocation
			space.put(nextFreeSpace+INT_SIZE,srcBytes);
			// Increment free space
			nextFreeSpace+=srcBytes.length+INT_SIZE; // The size requested plus an additional int			
		} else {
			// Otherwise, this is going to have to be a replacement
			replace(ref,src);
		}
	}

	/**
	 * We already have an item stored, we need to replace the value stored there.
	 * @param ref
	 * @param src
	 */
	protected void replace(int ref, String src) {
		if (ref < 0 || ref >= this.nextRef) {
			throw new RuntimeException("reference is out of range");
		}
		byte[] srcBytes = src.getBytes();
		if (getSize(ref) > srcBytes.length) {
			// easy case is that the new item is smaller than the original
			int offset = getAddress(ref);
			space.put(offset, srcBytes);
			// Update the size
			space.putInt(offset-INT_SIZE, srcBytes.length);
		} else {
			wastedSpace += getSize(ref);
			// New item is longer than the original, allocate anew
			setPointerAddress(ref,nextFreeSpace);
			space.putInt(nextFreeSpace, srcBytes.length);	// size of the allocation
			space.put(nextFreeSpace+INT_SIZE,srcBytes);
			// Increment free space
			nextFreeSpace+=srcBytes.length+INT_SIZE; // The size requested plus an additional int		
		}
	}
	
	/**
	 * The number of references allocated
	 * @return
	 */
	public int count() {
		return nextRef;
	}
	
	/**
	 * Store a string and return a reference to it
	 * @param ref
	 * @return
	 */
	public String fetch(int ref) {
		if (ref < 0 || ref >= this.nextRef) {
			throw new RuntimeException("reference is out of range");
		}
		int src = getAddress(ref);
		int size = getSize(ref);
		byte[] bytes = new byte[size];
		space.get(src, bytes);
		return new String(bytes);
	}
	
	/**
	 * The amount of space that can no longer be allocated
	 * (without some effort)
	 * @return
	 */
	public int getWastedSpace() {
		return wastedSpace;
	}
	
	public int getSpaceUsed() {
		return this.nextFreeSpace;
	}

	@Override
	public void close() throws IOException {
		saveHeader();		
	}
}
