package net.sodacan.core.serialize.kryo;

import java.time.Instant;
import java.util.HashMap;
import java.util.LinkedList;

import com.esotericsoftware.kryo.Kryo;

import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Message;
import net.sodacan.core.Route;
import net.sodacan.core.Serializer;
import net.sodacan.core.config.ActorMetadata;
import net.sodacan.core.serialize.SerializerFactory;
import net.sodacan.core.util.Pool;
import net.sodacan.core.util.PoolItemFactory;

/**
 * Factory that creates individual Serializers. Each actor will ask for a serializer.
 * The factory is free to hand out individual Serializers or a single shared serializer.
 * In any case, the factory will be getting requests from multiple threads and so should be sure to avoid
 * concurrency issues.
 * 
 * <p>Our constructor is created during Config. At this time, before we are accessed by Actors, we 
 * create a thread-safe pool of Kryo instances.</p>
 * 
 */
public class KryoSerializerFactory implements SerializerFactory, PoolItemFactory<Kryo> {
	private Config config;
	protected Class<? extends Message> messageClass;
	protected Pool<Kryo> pool;
	
	public KryoSerializerFactory(Config config) {
		super();
		this.config = config;
		messageClass = config.createMessage().getClass();
		 // Keep a pool of instances, don't forget to return after use
		pool = new Pool<>(10, this);
	}

	public Kryo createItem() {
		Message m99 = config.createMessage();
		Kryo kryo = new Kryo();
		kryo.register(Instant.class, 101);
		kryo.register(HashMap.class, 102);
		kryo.register(LinkedList.class, 103);
		kryo.register(ActorId.class, 202);
//		kryo.register(Message.class, 204);	// <<< Note
		kryo.register(m99.getClass(), 205);	// <<< Note
		kryo.register(m99.getMessageId().getClass(), 206);	// <<< Note
		kryo.register(Route.class, 207);	// <<< Note
		int appBase = 300;
		// Application-specific classes we need to register
		// must be Sorted
//		for (Class<?> genericClass : config.getActorMetadata().) {
//			// This should ignore duplicates
//			kryo.register(genericClass, ++appBase);
//		}
		return kryo;
	}
	
	/**
	 * Create a serializer for the specified actorId. This method is called from an individual Actor's thread.
	 * We don't modify any shared variables here but if we did, then syncing is needed.
	 */
	public Serializer create(ActorId actorId) {
		// Lookup metadata for the specified ActorId (actorType).
		String actorType = actorId.getType();
		ActorMetadata actorMetadata = config.getActorMetadata(actorType);
		Serializer serializer = new KryoSerializer(this, actorId, actorMetadata);
		return serializer;
	}

	public Config getConfig() {
		return config;
	}

	public Class<? extends Message> getMessageClass() {
		return messageClass;
	}
}
