package net.sodacan.core.serialize.kryo;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.lang.reflect.Field;
import java.nio.ByteBuffer;
import java.time.Instant;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Map.Entry;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.ByteBufferOutput;
import com.esotericsoftware.kryo.io.Input;
import com.google.gson.Gson;

import net.sodacan.core.Actor;
import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Message;
import net.sodacan.core.Route;
import net.sodacan.core.Serializer;
import net.sodacan.core.config.ActorMetadata;
import net.sodacan.core.serialize.gson.GsonSerializerFactory;
import net.sodacan.core.serialize.gson.GsonSerializer.ActorSave;

public class KryoSerializer implements Serializer {
	protected KryoSerializerFactory factory;
	protected ActorMetadata actorMetadata;
	protected ActorId actorId;
	
	public KryoSerializer(KryoSerializerFactory factory, ActorId actorId, ActorMetadata actorMetadata) {
		this.factory = factory;
		this.actorMetadata = actorMetadata;
		this.actorId = actorId;
	}

	@Override
	public byte[] serialize(Actor actor) {
		Kryo kryo = null;
		try {
			kryo = factory.pool.acquireItem();
			Map<String, Object> fieldValues = new HashMap<>();
			for (Entry<String,Field> entry : actorMetadata.getSaveFields().entrySet()) {
					Object value = entry.getValue().get(actor);
					if (value!=null) {	// Only save non-null fields
						fieldValues.put(entry.getKey(), value);
					}
			}
			ActorSave as = new ActorSave(actor.getActorId(), fieldValues);
			ByteBuffer bb = ByteBuffer.allocate(100);
			ByteBufferOutput output = new ByteBufferOutput(bb);
			kryo.writeObject(output, as);
			return output.toBytes();
		} catch (Exception e) {
			throw new RuntimeException("Error Serializing actor " + actor.getActorId().toString(), e);
		} finally {
			try {
				factory.pool.returnItem(kryo);
			} catch (InterruptedException e) {
			}
		}
	}

	@Override
	public void deserialize(byte[] source, Actor actor) {
		String actorType = actor.getActorId().getType();
		Kryo kryo = null;
		try {
			kryo = factory.pool.acquireItem();
			Input input = new Input(source);
			ActorSave as = kryo.readObject(input, ActorSave.class);
			for (Entry<String,Field> entry : factory.getConfig().getActorMetadata(actorType).getSaveFields().entrySet()) {
				Field field = entry.getValue();
				Object value = as.fields.get(entry.getKey());
				field.set(actor, value);
			}
		} catch (Exception e) {
			throw new RuntimeException("Error Deserializing actor " + actor.getActorId().toString(), e);
		} finally {
			try {
				factory.pool.returnItem(kryo);
			} catch (InterruptedException e) {
			}
		}
	}

	@Override
	public byte[] serialize(Message message) {
		Kryo kryo = null;
		try {
			kryo = factory.pool.acquireItem();
			ByteBuffer bb = ByteBuffer.allocate(100);
			ByteBufferOutput output = new ByteBufferOutput(bb);
			kryo.writeObject(output, message);
			return output.toBytes();
		} catch (Exception e) {
			throw new RuntimeException("Error Serializing message ", e);
		} finally {
			try {
				factory.pool.returnItem(kryo);
			} catch (InterruptedException e) {
			}
		}
	}

	@Override
	public Message deserialize(byte[] source) {
		Kryo kryo = null;
		try {
			kryo = factory.pool.acquireItem();
			Input input = new Input(source);
			Message m = kryo.readObject(new Input(source), factory.getMessageClass());
			return m;
		} catch (Exception e) {
			throw new RuntimeException("Error Deserializing message", e);
		} finally {
			try {
				factory.pool.returnItem(kryo);
			} catch (InterruptedException e) {
			}
		}
	}

	public static class ActorSave {
		ActorId actorId;
		Map<String, Object> fields;
		public ActorSave(ActorId actorId, Map<String, Object> fieldValues) {
			super();
			this.actorId = actorId;
			this.fields = fieldValues;
		}
		public ActorId getActorId() {
			return actorId;
		}
		public Map<String, Object> getFieldValues() {
			return fields;
		}
	}

}
