package net.sodacan.core.serialize.gson;

import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.InstanceCreator;
import com.google.gson.JsonArray;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.reflect.TypeToken;

import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Message;
import net.sodacan.core.MessageId;
import net.sodacan.core.Route;
import net.sodacan.core.Serializer;
import net.sodacan.core.Verb;
import net.sodacan.core.config.ActorMetadata;
import net.sodacan.core.message.DefaultMessageId;
import net.sodacan.core.serialize.SerializerFactory;
import net.sodacan.core.util.Pool;
import net.sodacan.core.util.PoolItemFactory;

/**
 * Factory that creates individual Serializers. Each actor will ask for a serializer.
 * The factory is free to hand out individual Serializers or a single shared serializer.
 * In any case, the factory will be getting requests from multiple threads and so should be sure to avoid
 * concurrency issues.
 * 
 * <p>Our constructor is created during Config. At this time, before we are accessed by Actors, we 
 * create a thread-safe pool of Gson instances.</p>
 * 
 */
public class GsonSerializerFactory implements SerializerFactory, PoolItemFactory<Gson> {
	private Config config;
	protected Class<? extends Message> messageClass;
	protected GsonBuilder builder;
	protected Pool<Gson> pool;
	
	public GsonSerializerFactory(Config config) {
		super();
		this.config = config;
		messageClass = config.createMessage().getClass();
		 // Keep a pool of instances, don't forget to return after use
		pool = new Pool<>(10, this);
	}

	public GsonBuilder getBuilderFor(ActorId actorId) {
		return builder;
	}

	public Gson createItem() {
		Type mapType = new TypeToken<Map<String, Object>>(){}.getType();
		return new GsonBuilder()
				.setPrettyPrinting()
				.registerTypeAdapter(Instant.class, new InstantTypeAdapter())
				.registerTypeAdapter(MessageId.class, new MessageIdAdapter())
				.registerTypeAdapter(Route.class, new RouteAdapter())
				.registerTypeAdapter(ActorId.class, new ActorIdAdapter())
				.registerTypeAdapter(mapType, new MapAdapter())
				.create();
	}
	
	/**
	 * Create a serializer for the specified actorId. This method is called from an individual Actor's thread.
	 * We don't modify any shared variables here but if we did, then syncing is needed.
	 */
	public Serializer create(ActorId actorId) {
		// Lookup metadata for the specified ActorId (actorType).
		String actorType = actorId.getType();
		ActorMetadata actorMetadata = config.getActorMetadata(actorType);
		Serializer serializer = new GsonSerializer(this, actorId, actorMetadata);
		return serializer;
	}

	public Config getConfig() {
		return config;
	}

	public Class<? extends Message> getMessageClass() {
		return messageClass;
	}

}
