package net.sodacan.core.serialize.gson;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import com.google.gson.Gson;

import net.sodacan.core.Actor;
import net.sodacan.core.ActorId;
import net.sodacan.core.Message;
import net.sodacan.core.Serializer;
import net.sodacan.core.config.ActorMetadata;

/**
 * Serialize and deserialize Actors and Messages to Json using Google's Gson.
 * We're instantiated during configuration so that the methods can be called
 * safely from any thread.
 * 
 */
public class GsonSerializer implements Serializer {
	protected GsonSerializerFactory factory;
	protected ActorMetadata actorMetadata;
	protected ActorId actorId;
	public GsonSerializer(GsonSerializerFactory factory, ActorId actorId, ActorMetadata actorMetadata) {
		this.factory = factory;
		this.actorMetadata = actorMetadata;
		this.actorId = actorId;
	}

	public static class ActorSave {
		ActorId actorId;
		Map<String, Object> fields;
		public ActorSave(ActorId actorId, Map<String, Object> fieldValues) {
			super();
			this.actorId = actorId;
			this.fields = fieldValues;
		}
		public ActorId getActorId() {
			return actorId;
		}
		public Map<String, Object> getFieldValues() {
			return fields;
		}
	}
	
	/**
	 * We are asked to serialize the fields of an actor, rather than the Actor as a whole.
	 * We serialize it as a map which is, in turn, taken care of in the MapAdapter
	 */
	@Override
	public byte[] serialize(Actor actor) {
		// Don't save if no fields to save
		if (actorMetadata.getSaveFields().isEmpty()) {
			return null;
		}
		Gson gson = null;
		try {
			gson = factory.pool.acquireItem();
			Map<String, Object> fieldValues = new HashMap<>();
			for (Entry<String,Field> entry : actorMetadata.getSaveFields().entrySet()) {
					Object value = entry.getValue().get(actor);
					if (value!=null) {	// Only save non-null fields
						fieldValues.put(entry.getKey(), value);
					}
			}
			ActorSave as = new ActorSave(actor.getActorId(), fieldValues);
			String json = gson.toJson(as);
			return json.getBytes();
		} catch (Exception e) {
			throw new RuntimeException("Error Serializing actor " + actor.getActorId().toString(), e);
		} finally {
			try {
				factory.pool.returnItem(gson);
			} catch (InterruptedException e) {
			}
		}
	}
	
	/**
	 * To deserialize an Actor requires two steps. First, deserialize a Map of the Actor's field values.
	 * Then populate the Actor's @Save fields from the deserialized Map. 
	 */
	@SuppressWarnings("unchecked")
	@Override
	public void deserialize(byte[] source, Actor actor) {
		Gson gson = null;
		String actorType = actor.getActorId().getType();
		try {
			gson = factory.pool.acquireItem();
			ActorSave as = gson.fromJson(new String(source), ActorSave.class);		
			for (Entry<String,Field> entry : factory.getConfig().getActorMetadata(actorType).getSaveFields().entrySet()) {
				Field field = entry.getValue();
				Object value = as.fields.get(entry.getKey());
				field.set(actor, value);
			}
		} catch (Exception e) {
			throw new RuntimeException("Error Deserializing actor " + actor.getActorId().toString(), e);
		} finally {
			try {
				factory.pool.returnItem(gson);
			} catch (InterruptedException e) {
			}
		}
	}

	@Override
	public byte[] serialize(Message message) {
		Gson gson = null;
		try {
			gson = factory.pool.acquireItem();
			String json = gson.toJson(message);
			return json.getBytes();
		} catch (Exception e) {
		} finally {
			try {
				factory.pool.returnItem(gson);
			} catch (InterruptedException e) {
			}
		}
		return null;
	}

	@Override
	public Message deserialize(byte[] source) {
		Gson gson = null;
		try {
			gson = factory.pool.acquireItem();
			
			return gson.fromJson(new String(source), factory.getMessageClass());
		} catch (Exception e) {
		} finally {
			try {
				factory.pool.returnItem(gson);
			} catch (InterruptedException e) {
			}
		}
		return null;
	}

}
