package net.sodacan.core.scheduler;

import java.io.Closeable;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import net.sodacan.core.ActorGroup;
import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Jug;
import net.sodacan.core.Scheduler;
import net.sodacan.core.actor.ActorEntry;
import net.sodacan.core.jug.AuditMessageLoad;
import net.sodacan.core.jug.CloseHostBoundActor;
import net.sodacan.core.jug.ErrorMessage;
import net.sodacan.core.jug.NormalMessage;
import net.sodacan.core.jug.StoppedActor;

public class DefaultScheduler extends Thread implements Scheduler, Closeable {
	private Config config;
	protected ActorGroup actorGroup;
	// Statistics
	private Statistics stats;
	private ThreadPoolExecutor pool;

	private BlockingQueue<Jug> messageQueue = null;
	private Map<ActorId, ActorEntry> actorEntries = new HashMap<>();
	
	/**
	 * Construct a default scheduler, one per ActorGroup
	 * @param config Configuration used by this scheduler
	 */
	public DefaultScheduler(Config config,  ActorGroup actorGroup) {
		this.config = config;
		this.actorGroup = actorGroup;
		// Thread pool
		int tpSize = config.getActorGroupThreads();
		if (tpSize>1) {
			pool = (ThreadPoolExecutor) Executors.newFixedThreadPool(tpSize);
		} else {
			pool = (ThreadPoolExecutor) Executors.newCachedThreadPool();
		}
		messageQueue = new ArrayBlockingQueue<>(config.getBackpressureLimit()*2);
		stats = new Statistics(actorGroup);
	}
	
	/**
	 * These are live statistics. You only need to call this method once. Calling individual methods will get you
	 * the current value.
	 */
	@Override
	public Statistics getStatistics() {
		return stats;
	}
	
	public ActorGroup getActorGroup() {
		return actorGroup;
	}

	/**
	 * Add a message to the work queue.
	 * First time through starts the thread.
	 * Executes in caller thread, not scheduler
	 */
	@Override
	public void addMessage(Jug jug) {
		messageQueue.add(jug);
		if (stats.running.compareAndSet(false, true)) {
			start();
		}
		increaseMessageLoad();
	}

	/**
	 * Add a new message to the work queue with backpressure.
	 */
	@Override
	public void addNewMessage(Jug jug) {
		// Apply Backpressure
		applyBackpressure();
		if (stats.closing.get()) {
			throw new RuntimeException("ActorGroup " + getActorGroup() +  " is closing down, no new messages");
		}
		addMessage(jug);
	}
	
	/**
	 * If there's too many messages in-flight, we're going to wait for a bit
	 */
	protected void applyBackpressure() {
		while (stats.getMessageLoad()>config.getBackpressureLimit()) {
			try {
				this.incrementSleepTime();
//				System.out.println("Applying Backpressure");
				Thread.sleep(config.getBackpressureWaitMs());
			} catch (InterruptedException e) {
			}
		}
	}

	@Override
	public void run() {
		while (!isInterrupted()) {
			Jug jug = null;
			try {
				jug = messageQueue.take();
			} catch (InterruptedException e) {
				break;
			}
			switch (jug)  {
				case AuditMessageLoad aml -> auditMessageLoad(aml);
				case StoppedActor sa -> stoppedActor(sa);
				case ErrorMessage em -> errorMessage(em);
				case NormalMessage nm -> normalMessage(nm);
				case CloseHostBoundActor chba -> closeHostBoundActor(chba);
				 default -> throw new RuntimeException("Invalid message received");
			};
		}
	}

	protected void normalMessage(NormalMessage nm) {
		increaseMessageCount();
		ActorId actorId = nm.getActorId();
		ActorEntry actorEntry = actorEntries.get(actorId);
		if (actorEntry==null) {
			actorEntry = new ActorEntry(config, this, actorId);
			actorEntries.put(actorId, actorEntry);
		}
		actorEntry.queueMessage(nm);
		actorEntry.startThread(pool);
	}
	/**
	 * Reactivate any actors with messages waiting.
	 */
	public void auditMessageLoad(AuditMessageLoad aml) {
		for (ActorEntry ae : actorEntries.values()) {
			int size = ae.getQueueSize();
			if (size>0) {
//				System.out.println("...Actor " + ae.getActorId() + " has " + size + " messages waiting" );
				ae.startThread(pool);
			}
		}
		decreaseMessageLoad();
	}

	public void errorMessage(ErrorMessage em) {
		System.err.println("Error from Actor " + em.getActorId());
		System.err.println(em.getException());
		decreaseMessageLoad();
	}
	
	/**
	 * HostBound Actors are always alive so just forward to the actor
	 * @param chba
	 */
	public void closeHostBoundActor(CloseHostBoundActor chba) {
		increaseMessageCount();
		ActorId actorId = chba.getActorId();
		ActorEntry actorEntry = actorEntries.get(actorId);
		if (actorEntry==null) {
			actorEntry = new ActorEntry(config, this, actorId);
			actorEntries.put(actorId, actorEntry);
		}
		actorEntry.queueMessage(chba);
		actorEntry.startThread(pool);
		
	}
	/**
	 * Decide if we're going to evict an ActorEntry and if so, do it.
	 * @param ae
	 */
	protected boolean evict(ActorEntry ae) {
		// We don't evict HostBound Actors
		if (ae.getActorId().getActorGroup() < 0) return false;
		if (ae.evictionProbability() < 0.5f) return false;
		actorEntries.remove(ae.getActorId());
		increaseEvictionCount();
		return true;
	}
	
	/**
	 * When an actorEntry says there are no more messages and thus has stopped, 
	 * see if there's actually more work to do. If so, start the thread back up.
	 * If not, consider removing the ActorEntry and Actor.
	 * By the time we get this message, the Actor should be fully passivated
	 * so that the Actor can be safely removed from memory.
	 * @param sa Sent from an actor that is done processing messages
	 */
	protected void stoppedActor(StoppedActor sa) {
		decreaseMessageLoad();
		ActorId actorId = sa.getActorId();
		ActorEntry ae = actorEntries.get(actorId);
		if (ae==null) return;
		if (ae.getQueueSize()>0) {
			ae.startThread(pool);
		} else {
			if (evict(ae)) {
//				System.out.println(">>>>>>Removing " + ae);
			}
		}
	}
	/**
	 * <p>Subjective check to see if messages are being processed.
	 * </p>
	 * $$$$ NOTE: This check must be done inside scheduling thread
	 * @return true if appears messages are bing processed since last time
	 * we were called
	 */
	protected boolean livelinessCheck() {
//		if (lastMessageLoad==getMessageLoad() 
//		&& lastMessageCount==getMessageCount() 
//		&& lastTotalQueueSize) {
//			return false;
//		}
//		lastMessageLoad = getMessageLoad();
//		lastMessageCount = getMessageCount();
		return true;
	}
	
	/**
	 * <p>In production, this is where we wait for an ActorGroup to settle down
	 * so we can move the ActorGroup to another host or otherwise change it's state.
	 * </p>
	 * <p>In addition to waiting for message load to go to zero, we also
	 * do a rough check to see if any messages are getting processed and if not,
	 * something is wrong.</p>
	 * 
	 */
	@Override
	public void waitForMessagesToFinish() {
		try {
			// When message load goes to zero, we return
			while (0!=stats.getMessageLoad()) {
				if (0>stats.getMessageLoad()) {
					throw new RuntimeException("Negative messageLoad, something is wrong");
				}
				if (!livelinessCheck()) {
					throw new RuntimeException("No message activity, something is wrong");					
				}
				System.out.println("Waiting for " + stats.getMessageLoad() + " messages to finish in actorGroup " + getActorGroup().getActorGroupNumber() );
				addNewMessage(new AuditMessageLoad());
				Thread.sleep(config.getShutdownWaitMs());
			}
			System.out.println("No more messages to process for actorGroup " + getActorGroup());
			
		} catch (InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	@Override
	public void close() {
		try {
			waitForMessagesToFinish();
			// No more messages allowed after this
			stats.closing.set(true);
			// Shut down the thread
			interrupt();
			pool.awaitTermination(1, TimeUnit.SECONDS);
			pool.shutdown();
			// Verify that there are no unprocessed message in an Actor's queue
			for (ActorEntry ae : actorEntries.values()) {
				if (ae.getQueueSize()!=0) {
					throw new RuntimeException("Non-empty queue in ActorEntry " + ae);
				}
			}
			// Poof, all actors are gone
			actorEntries.clear();
			System.out.println("Orderly shutdown complete for actorGroup: " + getActorGroup());
		} catch (Exception e) {
			throw new RuntimeException("Error closing Scheduler for ActorGroup " + this, e);
		}
	}

	@Override
	public Config getConfig() {
		return config;
	}

	@Override
	public void increaseMessageLoad() {
		int load = stats.messageLoad.incrementAndGet();
		if (load>stats.maxMessageLoad.get()) {
			stats.maxMessageLoad.set(load);
		}
	}

	@Override
	public void decreaseMessageLoad() {
		stats.messageLoad.decrementAndGet();		
	}

	@Override
	public void incrementSleepTime() {
		stats.totalSleepTime.addAndGet(getConfig().getBackpressureWaitMs());		
	}

	@Override
	public void increaseMessageCount() {
		stats.messageCount.incrementAndGet();
	}

	@Override
	public void increaseEvictionCount() {
		stats.evictionCount.incrementAndGet();		
	}
	
	@Override
	public void evictAll() {
		// TODO Auto-generated method stub
		
	}

	@Override
	public String toString() {
		if (actorGroup!=null) {
			return actorGroup.toString();
		} else {
			return "Standalone Scheduler";
		}
	}

	@Override
	public void incrementActorsCreated() {
		stats.totalActorsCreated.incrementAndGet();
	}

	
}
