package net.sodacan.core.persist;

import java.io.File;

import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Persister;
/**
 * This abstract class provides two services to concrete subclasses:
 * First, it stores the configuration and actorId of the Actor instance for which it applies.
 * Second, it provides a consistent file naming service in the case of a persister that uses
 * disk file(s). A Subclass is not required to use the naming service(s).
 */
public abstract class AbstractPersister implements Persister {
	private Config config;
	private ActorId actorId;
	
	public AbstractPersister( Config config, ActorId actorId) {
		this.config = config;
		this.actorId = actorId;
	}

	public Config getConfig() {
		return config;
	}

	public ActorId getActorId() {
		return actorId;
	}

	/**
	 * Files are organized by ActorGroup, type of actor, and actor id. The root folder is assumes to exist.
	 * This method creates the entire hierarchy of folders as needed and returns a file handle to the file.
	 * @return A File handle
	 */
	protected File makeFile(String extension) {
		File typeDir = makeTypeFolder();
		File file = new File(typeDir, actorId.getId() + extension);
//			System.out.println("File name: " + file.getAbsolutePath());
		return file;
	}

	/**
	 * This method return a file handle to folder assigned to the specific actorId.
	 * This method is used when the persister requires a folder into which possibly multiple files will be placed.
	 * This method creates the entire hierarchy of folders as needed and returns a file handle to the target folder.
	 * @return A File handle to the folder created
	 */
	protected File makeFolder() {
		File typeDir = makeTypeFolder();
		File folder = new File(typeDir, actorId.getId());
		if (!folder.exists()) {
			folder.mkdir();
		}
		return folder;
	}
	
	/**
	 * This method return a file handle to the ActorType directory within the specified ActorGroup directory within the specified Host directory
	 * within the specified root folder creating directories that do not exist.
	 * This method is already called by the makeFolder and makeFile methods. Therefore, it should normally not be called directly by a persister
	 * but might be overriden to provide a custom directory structure.
	 * @return A File handle to the ActorType directory
	 */
	protected File makeTypeFolder() {
		String rootDir = config.getRootDirectory();
		File root = new File(rootDir);
		File hostDir = new File(root, "H-" + config.getHostNumber());
		if (!hostDir.exists()) {
			hostDir.mkdir();
		}
		File actorGroupDir = new File(hostDir, "AG-" + actorId.getActorGroup());
		if (!actorGroupDir.exists()) {
			actorGroupDir.mkdir();
		}
		File typeDir = new File(actorGroupDir, actorId.getType());
		if (!typeDir.exists()) {
			typeDir.mkdir();
		}
		return typeDir;
	}
	
}
