package net.sodacan.core.message;

import java.io.PrintStream;
import java.time.Instant;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import net.sodacan.core.Actor;
import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Message;
import net.sodacan.core.MessageId;
import net.sodacan.core.Route;
import net.sodacan.core.Stage;
import net.sodacan.core.Verb;

/**
 * All Messages have a messageId and a routing stack. Many have a payload. 
 * A message can retain a history of where its been.
 */
public class DefaultMessage implements Message {	
	private MessageId messageId;
	private List<Route> history = new LinkedList<>();
	private boolean keepHistory = false;
	private Route currentRoute = null;
	// Message routes are in a stack: The top of the stack is the next route. 
	private LinkedList<Route> routes = new LinkedList<>();
	// A route used while building a message
	private Route buildRoute = null;

	private Map<String,Object> payload = new HashMap<>();

	public DefaultMessage() {
		
	}
	public DefaultMessage(MessageId messageId) {
		this.messageId = messageId;
	}
	
	public DefaultMessage(Message source) {
		this.messageId = source.getMessageId();
	}
	/**
	 *  Copy guts of provided message into this message but don't copy routes.
	 *  If keepHistory is false but there is lingering history from the original 
	 *  message, the history is NOT brought forward.
	 *  The messageId is brought forward.
	 *  Payload is brought forward.
	 *  This method is used to broadcast messages, with the same id, to multiple actors in parallel.
	 *  The single messageId allows for correlation of messages when needed.
	 * 
	 */
	public Message from(Message source) {
		this.messageId = source.getMessageId();
		this.keepHistory = source.isKeepHistory();
		if (keepHistory) {
			this.history = source.getHistory();
		}
		this.payload.putAll(source.getPayload());
		return this;
	}

	/**
	 * If nothing to do in the message, return false. Otherwise, check and throw exception if issues.
	 * @param config
	 * @return true if the message is valid
	 */
	public boolean validateMessage(Config config) {
		Route route = peekRoute();
		if (route==null) {
			throw new RuntimeException("Message has no routes, message not sent");
		}
		ActorId target = route.getTarget();
		if (target==null) {
			print(System.err);
			throw new RuntimeException("Message has no nextActorId, not sent");
		}
		if (target.getActorGroup()>config.getActorGroups()) {
			throw new RuntimeException("TargetActorId invalid ActorGroup, not sent");
		}
		if (route.getVerb()==null) {
			throw new RuntimeException("Verb missing, not sent");			
		}
		return true;
	}

	public MessageId getMessageId() {
		return messageId;
	}

	public Instant getTimestamp() {
		return messageId.getTimestamp();
	}
	
	public List<Route> getHistory() {
		List<Route> newHistory = new LinkedList<Route>();
		newHistory.addAll(history);
		return newHistory;
	}
	
	
	/**
	 * Get the current route
	 */
	@Override
	public Route getRoute() {
		return currentRoute;
	}

	@Override
	public ActorId getTarget() {
		if (currentRoute==null) {
			return null;
		}
		return currentRoute.getTarget();
	}

	@Override
	public Verb getVerb() {
		if (currentRoute==null) {
			return null;
		}
		return currentRoute.getVerb();
	}
	
	@Override
	public List<Route> getRoutes() {
		return routes;
	}

	@Override
	public ActorId getNextActorId() {
		Route route = peekRoute();
		if (route==null) return null;
		return peekRoute().getTarget();
	}

	/**
	 * Peek at the next verb or null if there's no more routes.
	 */
	@Override
	public Verb getNextVerb() {
		if (routes.size()==0) return null;
		return peekRoute().getVerb();
	}

	/**
	 * Peek at the next route or null if there's no more routes in the message.
	 * 
	 */
	public Route peekRoute() {
		if (routes.size()==0) return null;
		Route route = routes.peek();
		return route;
	}

	public Route peekTailRoute() {
		return routes.getLast();		
	}
		
	/**
	 * Popping a route sets the current route in the message.
	 */
	@Override
	public Route popRoute() {
		currentRoute = routes.pop();
		if (keepHistory) {
			history.add(currentRoute);
		}
		return currentRoute;
	}
	
	public void pushRoute(Route route) {
		routes.push(route);
	}

	@Override
	public Message keepHistory(boolean keepHistory) {
		this.keepHistory = keepHistory;
		if (keepHistory && history==null) {
			history = new LinkedList<>();
		}
		return this;
	}

	@Override
	public boolean isKeepHistory() {
		return keepHistory;
	}

	public Message trace(boolean trace) {
		return this;
	}
	
	// Copy of route of inbound message
	@Override
	public Message copyRoutesFrom(Message inbound) {
		routes.addAll(inbound.getRoutes());
		return this;
	}

	protected Route prepareBuildRoute() {
		if (buildRoute==null) {
			buildRoute = new Route();
		}
		return buildRoute;
	}
	
	// Route message to an actorId
	public Message ask(ActorId actorId) {
		prepareBuildRoute().setTarget(actorId);
		return this;
	}

	// Route message to an actorId defined in payload
	public Message ask(String key) {
		Object o = get(key);
		if (o==null) {
			throw new RuntimeException("The payload entry '" + key + "' is not found");
		}
		if (o instanceof ActorId actorId) {
			prepareBuildRoute().setTarget(actorId);
		} else {
			throw new RuntimeException("The payload entry '" + key + "' is not an ActorId");			
		}
		return this;
	}

	// Supply a verb to most recent route
	public Message to(Verb verb) {
		if (buildRoute==null) {
			throw new RuntimeException("copy is not setup");			
		}
		buildRoute.setVerb(verb);
		routes.addLast(buildRoute);
		buildRoute = null;
		return this;
	}

	public void toJson(PrintStream out) {
		out.println("no implemented");
	}

	@Override
	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append("Message Id: ");
		sb.append(getMessageId());
		sb.append("\n");
		if (getRoute()!=null) {
			sb.append("Current route: \n");
			sb.append(getRoute());
			sb.append("\n");
		}
		if (!routes.isEmpty()) {
			sb.append("Route Stack:\n");
			for (Route route : routes) {
				sb.append("  ");
				sb.append(route);
				sb.append("\n");
			}
		}
		if (!history.isEmpty()) {
			sb.append("History:\n");
			for (Route route : history) {
				sb.append("  ");
				sb.append(route);
				sb.append("\n");
			}
		}
		if (!payload.isEmpty()) {
			sb.append("Payload:\n");
			for (Entry<String, Object> entry : payload.entrySet()) {
				sb.append("  ");
				sb.append(entry.getKey());
				sb.append("=");
				sb.append(entry.getValue());
				sb.append('\n');
			}
		}
		return sb.toString();
	}

	@Override
	public void print(PrintStream out) {
		out.print(toString());
	}
	
	/**
	 * Get a fresh copy of the payload of this message
	 * @return
	 */
	@Override
	public Map<String, Object> getPayload() {
		return Map.copyOf(payload);
	}
		
	// Remove key from payload
	@Override
	public Message remove(String key) {
		payload.remove(key);
		return this;
	}
	/**
	 * Add an Info entry to payload, see convenience methods as well.
	 */
	@Override
	public Message put(String key, Object value) {
		payload.put(key, value);
		return this;
	}

	/**
	 * Put new String entry in payload. Convenience for a String
	 * @param key A string identifying the payload entry
	 * @param value The Payload value
	 * @return Message builder-style
	 */
	@Override
	public Message put(String key, String value) {
		payload.put(key, value);
		return this;
	}
	
	/**
	 * Put new Integer entry in payload. Convenience for an Integer
	 * @param key A string identifying the payload entry
	 * @param value The Payload value
	 * @return Message builder-style
	 */
	@Override
	public Message put(String key, Integer value) {
		payload.put(key, value);
		return this;
	}
	
	/**
	 * Put new entry in payload convenience for an ActorId
	 * @param key A string identifying the payload entry
	 * @param value The ActorId to add
	 * @return Message builder-style
	 */
	@Override
	public Message put(String key, ActorId value) {
		payload.put(key, value);
		return this;
	}	
	
	@Override
	public Object get(String key) {
		return payload.get(key);
	}

	@Override
	public <T> T get(String key, Class<T> clazz) {
		Object o = get(key);
		if (o==null) {
			throw new RuntimeException("Info entry '" + key + "' not found");			
		}
		if (o.getClass().isAssignableFrom(clazz)) {
			return (T) o;
		} else {
			throw new RuntimeException("Payload entry is not a " + clazz.getSimpleName() + " type");			
		}
	}
	
	/**
	 * Find the Info record the matching name (case insensitive) and return its String Contents
	 * @param key
	 * @return
	 */
	@Override
	public String getString(String key) {
		Object o = get(key);
		if (o==null) {
			throw new RuntimeException("Info entry '" + key + "' not found");			
		}
		if (o instanceof String si) {
			return si;
		} else {
			throw new RuntimeException("Payload entry is not a String type");
		}
	}

	@Override
	public int getInteger(String key) {
		Object o = get(key);
		if (o==null) {
			throw new RuntimeException("Info entry '" + key + "' not found");			
		}
		if (o instanceof Integer si) {
			return si;
		} else {
			throw new RuntimeException("Payload entry is not a Integer type");
		}
	}

	/**
	 * Find the Info record the matching name (case insensitive) and return its String Contents
	 * @param key
	 * @return
	 */
	@Override
	public ActorId getActorId(String key) {
		Object o = get(key);
		if (o==null) {
			throw new RuntimeException("Info entry '" + key + "' not found");			
		}
		if (o instanceof ActorId actorId) {
			return actorId;
		} else {
			throw new RuntimeException("Payload entry is not a ActorId type");
		}
	}
	
	@Override
	public boolean preprocess(Config config, Actor actor) {
		return true;
	}
	@Override
	public boolean postprocess(Config config, Actor actor) {
		return true;
	}
	@Override
	public Stage notImplemented(Config config, Actor actor) {
		return actor.getEmptyStage();
	}
	@Override
	public void terminal(Config config, Actor actor) {
	}	
}
	
