package net.sodacan.core.journal;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.Instant;

/**
 * <p>A journal entry suitable for writing/reading with nio. </p>
 * <p>Journal entries are stored and transmitted in BIG_ENDIAN byte order. 
 * Strings are stored as UTF-8.</p>
 */
public class JournalEntry {
	private long offset = -1; // Offset of this entry
	private long timestamp;
	private int flags = 0; // especially for "actor state" messages
	private int targetGroup = -1; // Extracted from targetActorId
	private int payloadSize = 0;
	private byte[] payload = null;
	
	private JournalEntry(long offset) {
		this.offset = offset;
	}
	public JournalEntry() {
		this.offset = 0;
	}
	public static JournalEntry fromByteBuffer(ByteBuffer bb) {
		bb.rewind();
		JournalEntry je = new JournalEntry();
		je.offset = bb.getLong();
		je.timestamp = bb.getLong();
		je.flags = bb.getInt();
		je.targetGroup = bb.getInt();
		je.payloadSize = bb.getInt();
		return je;
	}
	
	@Override
	public String toString() {
		StringBuffer sb = new StringBuffer();
		sb.append("Offset=");
		sb.append(this.offset);
		sb.append(", Timestamp=");
		sb.append(Instant.ofEpochMilli(this.timestamp));
		sb.append(", Flags=");
		sb.append(this.flags);
		sb.append(", Group=");
		sb.append(this.targetGroup);
		if (this.payload==null) {
			sb.append(", Payload skipped");
		} else {
			sb.append(", Payload=");
			sb.append(new String(this.payload,StandardCharsets.UTF_8));
		}
		return sb.toString();
	}


}
