package net.sodacan.core.host;

import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import net.sodacan.core.ActorGroup;
import net.sodacan.core.Config;
import net.sodacan.core.Coordinator;
import net.sodacan.core.Host;
import net.sodacan.core.Jug;
import net.sodacan.core.Message;
import net.sodacan.core.Route;
import net.sodacan.core.Serializer;
import net.sodacan.core.Stage;
import net.sodacan.core.coordinator.HostEntry;
import net.sodacan.core.coordinator.HostEntry.Mode;
import net.sodacan.core.jug.NormalMessage;
import net.sodacan.core.scheduler.Statistics;

/**
 * <p>A host starts out by hosting no actorGroups. The coordinator will tell the host
 * when it should:
 * <ul>
 * <li>Add a numbered actorGroup (the host will actually construct the actorGroup).</li>
 * <li>Tell a actorGroup to stop processing messages. This process can take time. </li>
 * <li>Tell a actorGroup to start processing messages. At this point, the actorGroup
 * will create a scheduler and be open to accepting new messages.</li>
 * <li></li>
 * <li></li>
 * </ul>
 * <p>The coordinator is not involved in message delivery, only to specify what each host
 * should do actorGroup-wise.
 * </p>
 *<p>All in-flight messages pass through the host where the actorGroup id of the target is determined and if
 * the actorGroup is active on this host, it will e forwarded, without intermediate queuing, to the actorGroup which will
 * then add it to the message queue of the target actor. If on the other hand, the host is not hosting the target actorGroup,
 * the message will be queued to a special static actorGroup and then to a special actor that will deal with communication
 * to the host where that target actor is "running".
 * </p>
 * <p>The only thing special about a special actorGroup is that it is locked to the host that created it. In other words,
 * a special actorGroup is only special to the host.
 * The special actorGroup is always active. The special actors are only special in that their purpose is to relay a message
 * to another remote host. The special actorGroup will have one actor for each host in the network. These actors are 
 * also used for sending actor replicas.
 * </p>
 */
public abstract class AbstractHost implements Host {
	private Config config;
	private int hostNumber;
	private boolean initialized = false;
	private Coordinator coordinator;
	protected Map<HostEntry,ActorGroup> activeActorGroups = new ConcurrentHashMap<>();

	public AbstractHost(Config config) {
		this.config = config;
		this.hostNumber = config.getHostNumber();
		this.coordinator = getConfig().getCoordinator();
	}
	
	@Override
	public Config getConfig() {
		return config;
	}

	@Override
	public int getHostNumber() {
		return hostNumber;
	}

	@Override
	public Map<Integer, Statistics> getActorGroupStatistics() {
		Map<Integer, Statistics> stats = new HashMap<>();
		for (Entry<HostEntry,ActorGroup> entry : activeActorGroups.entrySet()) {
			stats.put(entry.getValue().getActorGroupNumber(), entry.getValue().getScheduler().getStatistics());
		}
		return stats;
	}
	
	public boolean isInitialized() {
		return initialized;
	}

	@Override
	public Coordinator getCoordinator() {
		return coordinator;
	}

	public void setInitialized(boolean initialized) {
		this.initialized = initialized;
	}

	/**
	 * Get the updated configuration from coordinator and add/remove actor groups as
	 * needed.
	 */
	@Override
	public void update() {
		// Get the list of desired actorGroups
		List<HostEntry> actorGroups = coordinator.getActorGroupsForHost();
		// Also create a special ActorGroup bound to the host. The 
		// coordinator shouldn't know about this group, be we do.
		actorGroups.add(new HostEntry(config.getHostNumber(),-config.getHostNumber(),Mode.Host));

		
		// If any current ActorGroup has to go, make it gone
		for (Entry<HostEntry,ActorGroup> he: activeActorGroups.entrySet()) {
			// If not in the new list, close it down, remove from active 
			if (!actorGroups.contains(he.getKey())) {
				try {
					he.getValue().close();
				} catch (Exception e) {
					throw new RuntimeException("Unable to close ActorGroup " + he.getValue() ,e);
				}
				activeActorGroups.remove(he.getKey());
			}
		}
		// And for any new actorGroup not in active list, add it.
		for (HostEntry he : actorGroups) {
			if (!activeActorGroups.containsKey(he)) {
				ActorGroup ag = config.createActorGroup(he.getActorGroup());
				ag.setHost(this);
				activeActorGroups.put(he, ag);
			}
		}
	}
	
	@Override
	public Set<HostEntry> getHostEntries() {
		return this.activeActorGroups.keySet();
	}

	@Override
	public ActorGroup getActorGroup(HostEntry hostEntry) {
		return activeActorGroups.get(hostEntry);
	}

	@Override
	public Set<ActorGroup> getActorGroups() {
		Set<HostEntry> hes = getHostEntries();
		Set<ActorGroup> ags = new HashSet<>();
		for (HostEntry he : hes) {
			ags.add(getActorGroup(he));
		}
		return ags;
	}

	/**
	 * Close the host by closing the ActorGroups.
	 * ActorGroup &lt; 0 precedes normal actor groups.
	 */
	@Override
	public void close() throws IOException {
		// ActorGroup < 0 only
		for (ActorGroup ag : getActorGroups()) {
			if (ag.getActorGroupNumber()<0) {
				ag.close();
			}
		}
		// All but ActorGroup > 0 only
		for (ActorGroup ag : getActorGroups()) {
			if (ag.getActorGroupNumber()>0) {
				ag.close();
			}
		}
		activeActorGroups.clear();
	}

	protected void replicate(int host, NormalMessage nm) {
		System.out.println("Message " + nm + " replicated to host " + host);
	}

	/**
	 * Figure our where the host is and send this message there
	 * @param hostNumber
	 * @param nm A normal mesage
	 */
	protected void forwardToHost(int hostNumber, NormalMessage nm) {
		
	}
	
	/**
	 * Send this message to live actor and if this is the live ActorGroup, 
	 * copy it to other specified host(s)
	 */
	@Override
	public boolean send(Jug jug) {
		return switch(jug) {
			case NormalMessage nm -> sendNormalMessage(nm);
			default -> throw new RuntimeException("Invalid Jug receied by host");
		};
	}
	
	protected boolean sendNormalMessage(NormalMessage nm) {
		initialize();
		int agNumber = nm.getActorId().getActorGroup();
		HostEntry he;
		if (agNumber<0) {
			he = new HostEntry(getHostNumber(), agNumber, Mode.Host);
		} else {
			he = new HostEntry(getHostNumber(), agNumber, Mode.Live);						
		}
		ActorGroup ag = activeActorGroups.get(he);
		// Do we hold the live (or host) version of this actorGroup?
		if (ag!=null){
			// Yes, so add it for processing here
			ag.addMessage(nm);
			// And also, send it to others for replication
			for (Integer replicationHost : coordinator.getHostsForActorGroup(agNumber)) {
				if (replicationHost!=agNumber) {
					replicate(replicationHost, nm);
				}
			}
		} else {
			// No, must forward to proper host
			forwardToHost(coordinator.getLiveHostForActorGroup(agNumber), nm);	
		}
		return true;		
	}

	/**
	 * This method is only useful for testing when it is expected that the message load will drop to zero.
	 */
	public void waitForMessagesToFinish() {
		for (Entry<HostEntry, ActorGroup> entry : activeActorGroups.entrySet()) {
			if (entry.getValue().getActorGroupNumber()>0) {		// Ignore HostBound actors, they have a different lifecycle
				entry.getValue().getScheduler().waitForMessagesToFinish();
			}
		}
	}

}
