package net.sodacan.core.actorgroup;


import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;
import java.util.concurrent.CountDownLatch;

import net.sodacan.core.Actor;
import net.sodacan.core.Actor.Builtin;
import net.sodacan.core.ActorGroup;
import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Host;
import net.sodacan.core.Jug;
import net.sodacan.core.Message;
import net.sodacan.core.Scheduler;
import net.sodacan.core.jug.CloseHostBoundActor;
import net.sodacan.core.jug.NormalMessage;

public abstract class AbstractActorGroup implements ActorGroup {
	private Config config;
	private int actorGroupNumber;
	private Scheduler scheduler;
	protected Host host = null;
	// Used to coordinate arrival of Builtin.Closed message for a HostBound actorGroup.
	protected CountDownLatch closeLatch = null;
	
	public AbstractActorGroup( Config config, int actorGroupNumber) {
		this.config = config;
		this.actorGroupNumber = actorGroupNumber;
		// Verify that actorGroup number is valid, positive numbers are actorgroup number,
		// negative numbers are host numbers, which are unlimited
		if ((actorGroupNumber > 0 && actorGroupNumber > config.getActorGroups()) 
			|| actorGroupNumber == 0) {
			throw new RuntimeException("actorGroup number (" + actorGroupNumber + ") out of range");
		}
		// Need a scheduler
		scheduler = config.createScheduler(this);
		this.host = config.getHost();
//		System.out.println("Actor group created: " + this);
		// Special handling for hostBound. Send a start message to the
		// host-bound actors to get things started.
		if (actorGroupNumber<0) {
			for (Entry<String, Class<? extends Actor>> entry  : config.getActorTypes().entrySet()) {
				String actorType = entry.getKey();
				if (actorType.startsWith("$")) {
					ActorId actorId = new ActorId(entry.getKey(), -config.getHostNumber(), "host");
					sendStartMessage(actorId);
				}
			}
		}
	}

	protected void sendStartMessage(ActorId actorId) {
		Message m = config.createMessage();
		m.ask(actorId).to(Builtin.Start);
		byte [] bytes = config.getSerializerFactory().create(actorId).serialize(m);
		NormalMessage nm = new NormalMessage(actorId, bytes);
		addMessage(nm);
		
	}
	@Override
	public void setHost(Host host) {
		this.host = host;
	}


	@Override
	public Scheduler getScheduler() {
		return scheduler;
	}

	/**
	 * This works in two ways: For normal ActorGroups, close waits for the actor to finish processing messages.
	 * For Host-Bound Actors, we need to exchange close and closed messages to ensure each is properly shut down.
	 */
	@Override
	public void close() throws IOException, RuntimeException {
		// Send a close message to each actor that needs to close.
		// once we get a response back from each one, we can proceed
		if (actorGroupNumber<0) {
			// Build a list actorIds that need closing
			List<ActorId> closeActors = new LinkedList<>();
			for (Entry<String, Class<? extends Actor>> entry  : config.getActorTypes().entrySet()) {
				String actorType = entry.getKey();
				if (actorType.startsWith("$")) {
					ActorId actorId = new ActorId(entry.getKey(), -config.getHostNumber(), "host");
					closeActors.add(actorId);
				}
			}
//			System.out.println("Setting up CloseLatch: " + closeActors.size());
			// Setup closeLatch
			closeLatch = new CountDownLatch(closeActors.size());
			// Send a close to each one
			for (ActorId actorId : closeActors) {
				addMessage(new CloseHostBoundActor(actorId, closeLatch));
			}
			try {
			closeLatch.await();
			} catch (Exception e) {
				throw new RuntimeException("Error creating close latch", e);
			}
			// Fall through to do the rest of the closing process
		}
		scheduler.close();
		System.out.println("Actor group closed: " + this);
	}

	public Config getConfig() {
		return config;
	}
	public int getActorGroupNumber() {
		return actorGroupNumber;
	}

	public void increaseMessageLoad() {
		scheduler.increaseMessageLoad();
	}

	public void reduceMessageLoad() {
		scheduler.decreaseMessageLoad();
	}

	@Override
	public String toString() {
		return "ActorGroupNumber=" + getActorGroupNumber();
	}

	/**
	 * <P>Queue this message for processing by scheduler
	 * </p>
	 */
	@Override
	public void addMessage(Jug jug) {
//		System.out.println("In actorGroup");
//		message.print(System.out);
		// Check the actorGroup for the target actor
		getScheduler().addNewMessage(jug);
	}
	
}
