package net.sodacan.core.actor;

import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;

import net.sodacan.core.ActorId;
import net.sodacan.core.ActorIdFactory;
import net.sodacan.core.Config;

/**
 * Create a sequential ActorId for testing. The counter used in this Factory 
 * would have to be persisted in order to be useful in production. Also, the number is incremented in the 
 * ActorGroup of the actor creating the actorId so it is very likely that there would be an id collision
 * if two actors, each in a different actorGroup, created an id was sent to the same ActorGroup. UUID is much safer
 * but about three times slower than this class.
 */
public class SequentialActorIdFactory implements ActorIdFactory {
	Config config;
	private AtomicLong seq = new AtomicLong(0);
	public SequentialActorIdFactory(Config config) {
		this.config = config;
	}

	@Override
	public ActorId create(String actorType) {
		if (config.getActorClass(actorType)==null) {
			throw new RuntimeException("Invalid ActorType: " + actorType);
		}
		String id = Long.toString(seq.incrementAndGet());
		int actorGroup = config.getActorGroupAssigner().assign(actorType, id);
		// Override id if it's a HostBoound Actor.
		if (actorGroup<0) {
			id = "host";
		}
		return new ActorId(actorType,actorGroup,id);
	}
}
