package net.sodacan.core.actor;

import java.time.Duration;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.atomic.AtomicBoolean;

import net.sodacan.core.Actor;
import net.sodacan.core.ActorGroup;
import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Jug;
import net.sodacan.core.Message;
import net.sodacan.core.Persister;
import net.sodacan.core.Route;
import net.sodacan.core.Scheduler;
import net.sodacan.core.Serializer;
import net.sodacan.core.Stage;
import net.sodacan.core.jug.CloseHostBoundActor;
import net.sodacan.core.jug.ErrorMessage;
import net.sodacan.core.jug.NormalMessage;
import net.sodacan.core.jug.StoppedActor;
import net.sodacan.core.persist.PersisterFactory;

/**
 * <p>The scheduler for a actorGroup manages a map of ActorEntries. 
 * Each entry wraps a single actor instance. This entry
 * is what contains the "Runnable" entity for purposes of executing
 * the actor's processMessage method as messages arrive.</p>
 */	
public class ActorEntry extends Thread implements Runnable {
	private AtomicBoolean running = new AtomicBoolean(false);
	private boolean initialized = false;
//	private boolean errorSkip = false;
	protected ActorId actorId = null;
	protected Actor actor = null;
	private BlockingQueue<Jug> queue = new LinkedBlockingQueue<>();
	protected Scheduler scheduler;
	protected Config config;
	protected long usage = 0;
	protected boolean evicting = false;
	protected Serializer serializer;
	public ActorId getActorId() {
		return actorId;
	}

	public long getUsage() {
		return usage;
	}
	/**
	 * Kick off the thread if its not running
	 */
	public void startThread(ThreadPoolExecutor pool) {
		if (running.compareAndSet(false, true)) {
			pool.execute(this);
		}
	}
	
	public void stopThread() {
		if (running.compareAndSet(true, false)) {
//			interrupt();
		}
	}

	/**
	 * <p>Construct a new ActorEntry and instantiate and actor contained within
	 * it using a configured actor factory. The only thing an actor "knows" after
	 * creation is its ActorId. Additional saved state will arrive shortly
	 * after creation, if needed. This is necessary to maintain the spirit of
	 * the actor model in Sodacan.</p>
	 * @param config 
	 * @param scheduler
	 * @param actorId
	 */
	public ActorEntry(Config config, Scheduler scheduler, ActorId actorId) {
		this.config = config;
		this.scheduler = scheduler;
		this.actorId = actorId;
	}
	
	/**
	 * Callable from any thread to add a message to this queue.
	 * Usually called from the scheduler. This queue has no limits. Rather, the message load is used to
	 * stall new messages before they get to this point.
	 * @param jug The Jug (message) to be queued
	 */
	public void queueMessage(Jug jug) {
		queue.add(jug);
//		scheduler.increaseMessageLoad();
	}
	
	protected void initialize() throws Exception {
		if (initialized) return;
		// The Actor instance needs to exist now (initializing an actor could take time.
		// The time is charged to the actor, a good thing.)
		if (actor==null) {
			actor = config.createActor(actorId);
			actor.setActorGroup(scheduler.getActorGroup());
			scheduler.incrementActorsCreated();
			actor.restore();

		}
		initialized = true;
	}
	private int lingerCount = 0;
	/**
	 * If return true, then try to grab a new message, otherwise, get out
	 * 
	 * @return 
	 * @throws InterruptedException 
	 */
	protected boolean linger() throws InterruptedException {
		if (evictionProbability()>0.1f) {
			return false;
		}
		if (++lingerCount>2) {
			lingerCount=0;
			return false;			
		}
		Thread.sleep(Duration.ofMillis(2));
		return true;
	}

	@Override
	public void run() {
		// Process messages until interrupted or the queue is empty,
		// Exceptions thrown are handled within this loop
		while (!isInterrupted()) {
			Jug jug = queue.poll();
			boolean more = switch (jug) {
				case null -> emptyQueue();
				case CloseHostBoundActor chba -> closeHostBoundActor(chba);
				case NormalMessage nm -> normalMessage(nm);
				default -> throw new IllegalArgumentException("Unexpected value: " + jug);
			};
			if (!more) break;
		}
		scheduler.increaseMessageLoad();	// Holds the actor 'til this is done
		actor.exitingThread();
		scheduler.decreaseMessageLoad();
		// Remember that we're no longer running
		stopThread();
	}
	/**
	 * 
	 * @return
	 */
	protected boolean emptyQueue() {
		try {
			// Linger for a bit before bailing
			if (linger()) {
				return true;
			}
			// Tell Scheduler that we are stopped
			scheduler.addMessage(new StoppedActor(actorId));
			return false;
		} catch (Exception e) {
			throw new RuntimeException("Error in creating StoppedActor message", e);
		}
	}
	/**
	 * Once this actor is closed, we can countdown to allow the ActorGroup to close
	 * @param chba
	 * @return
	 */
	protected boolean closeHostBoundActor(CloseHostBoundActor chba) {
		actor.close();
		chba.countDown();
		scheduler.decreaseMessageLoad();
		return false;
	}
	
	protected boolean normalMessage(NormalMessage nm) {
		try {
			initialize();
			actor.callProcessMessage(nm);
			scheduler.decreaseMessageLoad();
		} catch (Throwable e) {
			System.err.println("Exception in actor " + actorId);
			scheduler.addMessage(new ErrorMessage(actorId, nm.getMessage(), e));
		}
		return true;
	}
	
	public int getQueueSize() {
		return queue.size();
	}

	/**
	 * An actor is subject to eviction at any time. However, the lower this value, the less
	 * likely is will be chosen for eviction.
	 */
	public float evictionProbability() {
		if (actor==null) return 1.0f;
		return (actor.evictionProbability());
	}

	@Override
	public String toString() {
		return "Entry for " + actorId.toString();
	}

}

