package net.sodacan.core.actor;

import java.util.Map;

import net.sodacan.core.Actor;
import net.sodacan.core.ActorGroup;
import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Jug;
import net.sodacan.core.Message;
import net.sodacan.core.Persister;
import net.sodacan.core.Route;
import net.sodacan.core.Serializer;
import net.sodacan.core.Stage;
import net.sodacan.core.jug.NormalMessage;
import net.sodacan.core.scheduler.Statistics;

public abstract class AbstractActor implements Actor {

	private ActorId actorId;
	private Config config;
	private ActorGroup actorGroup;
	private Serializer serializer;
	private Persister persister;
	
	public static Stage emptyStage = new Stage();
	
	public AbstractActor(Config config, ActorId actorId) {
		this.config = config;
		this.actorId = actorId;
		this.serializer = config.getSerializerFactory().create(actorId);
		this.persister = config.getPersisterFactory().create(actorId);
	}
	
	/**
	 * An actor is subject to eviction at any time. However, the lower this value, the less
	 * likely is will be chosen for eviction.
	 */
	@Override
	public float evictionProbability() {
		return 0.5f;
	}

	public void callProcessMessage(NormalMessage nm ) {
		// It's time to deserialize the message
		Message message = serializer.deserialize(nm.getMessage());
		// Let the message know that it is about to be processed by this Actor
		message.preprocess(config, this);
		message.popRoute();
		// process message
		Stage stage = processMessage(message);
		// Reduce the overall message load count after the message is processed
		handleStage(stage);
	}
	
	/**
	 * Process the stage - we're still in the Actor's run thread.
	 * Should be forwarded to the Host in case ActorGroup has changed.
	 * @param stage
	 */
	protected void handleStage(Stage stage) {
		for (Message msg : stage.getMessages()) {
			send(msg);
		}
	}

	@Override
	public void send(Message m) {
		if (m.peekRoute()==null) {
			m.terminal(config, this);
		} else {
			if (m.postprocess(config, this)) {
				Route route = m.peekRoute();
				if (route!=null) {
					Serializer serializer = config.getSerializerFactory().create(route.getTarget());
					Jug jug = new NormalMessage(route.getTarget(),serializer.serialize(m));
					config.getHost().send(jug);
				}
			}
		}
	}
	
	/**
	 * Default behavior for an actor is to do nothing
	 */
	@Override
	public Stage processMessage(Message message) {
		return message.notImplemented(config, this);
	}
	
	/**
	 * Make a map of actorGroup statistics. We have to ask the host for this.
	 * @return
	 */
	public Map<Integer, Statistics> getLiveActorGroupStatistics() {
		return config.getHost().getActorGroupStatistics();
	}
	
	/**
	 * Concrete actors that are host-bound and have resources
	 * should override this
	 */
	@Override
	public void close() { }

	/**
	 * Save state of actor in two steps: Serialize the content and then write the resulting byte array and then write it out
	 */
	@Override
	public void save() {
		byte[] ba = serializer.serialize(this);
		if (ba!=null) {
			persister.write("", ba);
		}
	}
	
	@Override
	public void restore() {
		byte[] ba = persister.read("");
		if (ba!=null) {
			serializer.deserialize(ba, this);
		}
	}

	/**
	 * The default saves the state of the actor
	 */
	@Override
	public void exitingThread() {
		save();
	}


	/**
	 * TODO Put a generic reject message into the stage. The reject will be returned 
	 * to the Actor that sent the message.
	 * @param message The problematic message
	 * @return The resulting stage
	 */
	protected Stage reject(Message message, String reason) {
		Stage stage = new Stage();
//		stage.add(message.reject(reason));
		return stage;
	}

	/**
	 * <p>Empty the list of staged messages and restore the state of the actor.
	 */
	protected void rollback() {
		
	}

	@Override
	public ActorId getActorId() {
		return actorId;
	}

	@Override
	public Config getConfig() {
		return config;
	}

	@Override
	public ActorGroup getActorGroup() {
		return actorGroup;
	}

	public void setActorGroup(ActorGroup actorGroup) {
		this.actorGroup = actorGroup;
	}

	@Override
	public Stage getEmptyStage() {
		return emptyStage;
	}

}
