package net.sodacan.core;

import java.util.LinkedList;
import java.util.List;

/**
 * <p>While an Actor processes a message, it populates this object with messages to be sent to other actors upon completion of processing the message.
 * The stage is also where, formally, the actor can say: "save the state of the object." That means all internal and external affects of an actor processing a single message
 * are encapsulated in the Stage. One could say, "If it's not in the Stage, it didn't happen." But this is only partially true for certain actors.
 * For those actors that are themselves the end-point of an action, then the "side effect" is the goal and the stage, at most, would contain 
 * a confirmation message. For example, an Actor that actually performs some kind of IO, like writing to a communication channel or actually turning a light on.
 * </p>
 * <p>The stage supports the commit/rollback semantics that many business transactions need. But the stage has no real behavior, it's just a container
 * used by the internal mechanisms to get messages to where they need to go.
 * </p>
 * <p>A Stage object has no concurrency requirements.</p>
 */
public class Stage {
	private List<Message> messages = new LinkedList<>();
	public Stage() {
		
	}
	
	public Message add(Message message) {
		messages.add(message);
		return message;
	}
	
	public List<Message> getMessages() {
		return messages;
	}

	/**
	 * <p>Send the message on to its next route. If there is no next route in the message, then we do nothing.</p>
	 * @param config
	 * @param inbound
	 */
	public Stage forward(Config config, Message inbound) {
		if (inbound.getNextActorId()!=null) {
			add(config.createMessage())
			.from(inbound)
			.copyRoutesFrom(inbound);
		}
		return this;
	}
}
