package net.sodacan.core;
/**
 * <p>A Serializer is concerned with serializing and deserializing Actors and Messages.
 * A Serializer typically requires Actor and Message class metadata which takes time to collect and parse. For this reason,
 * A Serializer is instantiated for each concrete Actor class.</p>
 * <p>When serialization is called depends on the life-cycle of an Actor. In summary, an Actor's state is restored when the Actor
 * is instantiated in response to receiving its first message. It will be saved either at the conclusion of processing a message or at another time
 * depending on that Actor's semantics. 
 * </p>
 * <p>A Message is serialized as soon as it is "sent" by an Actor and deserialized by the receiving Actor.</p>
 * <p>In all cases the serialization occurs in the individual Actor's thread
 *  which means there are usually no concurrency issues except that the 
 * Actor and Message metadata should be static by the time any serialization occurs.</p>
 * <p>Serialization must be to or from a byte array.</p>
 */
public interface Serializer {

	/**
	 * Serialize the @Save fields of an Actor
	 * @param actor
	 * @return
	 */
	public byte[] serialize(Actor actor);

	/**
	 * Deserialize the @Save fields of an Actor into an existing actor.  
	 * @param source
	 * @param actor
	 */
	public void deserialize(byte[] source, Actor actor);

	/**
	 * Serialize a Message to a byte array
	 * @param message
	 * @return
	 */
	public byte[] serialize(Message message);

	/**
	 * Deserialize a byte array into a new Message.
	 * @param source
	 * @return A newly constructed Message
	 */
	public Message deserialize(byte[] source);
	
}
