package net.sodacan.core;

/**
 * <p>Implementations of this interface save and restore a serialized form of an Actor. 
 * One Persister per Actor. However, persisters are free to consolidate resource usage but must 
 * be aware that doing so can encounter concurrency issues.
 * </p>
 */
public interface Persister {

	/**
	 * The framework calls the begin method when it is about to execute what might turn out to be a series of 
	 * calls to this persister that should be considered as a single transaction.
	 */
	public void begin();

	/**
	 * This method is called when the framework has concluded a series of writes that could be rolled back in case of an error.
	 */
	public void commit();

	/**
	 * When a series of writes should be rolled back: not written in the first place, replaced with a previous version, etc.
	 */
	public void rollback();
	
	/**
	 * The save method is called by the framework when an actor's state has been updated. They key provided is unique within an actor
	 * instance. The ActorId was provided during construction of this persister.
	 * @param key
	 * @param bytes
	 */
	public void write(String key, byte[] bytes);

	/**
	 * This method is called by the framework when an actor's processMessage method is about to be called
	 * for the first time after the Actor has been instantiated.
	 * @param key
	 */
	public byte[] read(String key);
	
}
