package net.sodacan.core;

import java.io.PrintStream;
import java.time.Instant;
import java.util.List;
import java.util.Map;


/**
 * <h2>Overview</h2>
 * <p>
 * Messages are used between Actors and therefore are typically declared as independent
 * top-level classes. Most Messages in Sodacan are just containers for some important capabilities:
 * </p>
 * <ul>
 * <li>MessageId - can be thought of as a transaction Id. It can group together any number of steps in a business process. And the messageId is also 
 * used as a correlation Id during parallel fan-out fan-in processing.</li>
 * <li>Routing stack - where the message is headed. The top-of-stack refers to the next Actor that the message is routed to.</li>
 * <li>Current route - If the message is currently being processed by an Actor, this primarily contains the Verb (the ask) for what it should do,</li>
 * <li>Routing History - When a route is popped off of the routing stack, it is added to the routing history.</li>
 * <li>Payload - The payload carries any number of primitive or complex data organized as key-value pairs. As a message travels from actor-to-actor,
 * the payload can accumulate data relevant to the transaction.</li>
 * </ul>
 * <p>A Message object is what the Actor programmer normally sees. However, when a Message is in transit, it is called a Jug which consists of
 * the serialized content of the message with the ActorId of the immediate destination of the message in plain-text.</p>
 * <h2>Message Lifecycle</h2>
 * <p>A message has a specific beginning and ending. A message travels from Actor to Actor. A message is created by the configuration object by
 * some Actor. Normally, the message exists until it explicitly "consumed". Messages can also be "held" by an actor although this should only occur in
 * specific cases. While a message is traveling between actors, most of its contents are serialized into a Jug making it hard to analyze. 
 * This usually isn't an issue because Actors always see messages in their deserialized form. 
 * </p>
 * <P>Note: All of the following occurs in the Actor's thread. While Sodacan processes a message, it ensures that the message makes it back out 
 * the other side of the Actor's processMessage method. Technically, the message itself is immutable but the contents of the message are used to create one or more messages.
 * So, if message A is being processed, then Message A should appear among the messages in the "Stage". Any other Messages in the Stage will have been 
 * created by the Actor. 
 * </p>
 * <p>Now, to complicate matters, a Message can be "split". This means that a Message inbound to an actor can be cloned into more than one output 
 * messages. The rule above still applies, only there is more than one copy of the message in the Stage. 
 * Each of the clones will have the same MessageId. This is done to allow a single "business flow" to operate in parallel for one or 
 * more steps in its lifecycle. Split messages can and likely will travel to different Hosts. A split like this normally forward the initial message
 * and forwarding it to the Actor that will collect the results and of course zero or more clones that will end up at the same collecting Actor.
 * Usually, the clones will end their lifecycle in the collecting Actor. Once the collecting Actor receives the expected number of 
 * results from each clone, it can forward on the original message to the next step(s).
 * </p>
 * <p>A message is officially done when it is placed on the stage with an empty Routing Queue, ie there is no more Steps in the process. </p>
 * <h2>Observability</h2>
 * <P>Because a message has a unique id, it can be tracked although this 
 * is usually not the case in production. </p>
 * <p>Messages provide a natural transport for certain trace requirements. A Sodacan message represents a business transaction typically carrying with is
 * much of the data associated with that transaction. It carries with it the Actors that it will visit and the Actors that it has visited.</p>
 * <p>During its lifetime a message can travel to many Actors.</p>
 * <p>When a message is created it can start with an ActorId already in its routing stack. That Actor can collect some or all Messages
 * for analysis. For example, the default messageId generator in part uses a UTC Instant. If the final route in a messages life is to an actor that
 * measures the total elapsed time of a message.</p>
 * <p>Another approach is to Subclass the DefaultMessage class in which case the subclass can do fine-grained tracing to any mechanism desired.
 * In this case, a message can keep track of each step in its journey incling the elapsed time used in each step. Also, Sodacan calls additional method stub in a Message
 * allowing further traceability.</p>
 */
public interface Message {
	public MessageId getMessageId();
	public Instant getTimestamp();
	public ActorId getNextActorId();
	public Verb getNextVerb();

	/**
	 * Return a copy of the message routing history
	 * @return
	 */
	public List<Route> getHistory();
	public Message keepHistory(boolean keepHistory);
	public boolean isKeepHistory();

	/**
	 * Get the current route. This is the route that gave rise to this call to processMessage
	 * @return The current route
	 */
	public Route getRoute();
	/**
	 * Get the target actorId of the current route. This is the route that gave rise to this call to processMessage
	 * @return The current target actorId
	 */
	public ActorId getTarget();
	/**
	 * Get the verb of the current route. This is the route that gave rise to this call to processMessage
	 * @return The current verb
	 */
	public Verb getVerb();
	
	public List<Route> getRoutes();
	public Route peekRoute();

	public Route popRoute();
	public void print(PrintStream out);
	
	// Copy of route of inbound message
	public Message copyRoutesFrom(Message inbound);

	// Copy guts of provided message to this message  but don't copy routes
	public Message from(Message source);
	// Add a new route to the route stack
	public Message ask(ActorId actorId);
	public Message ask(String key);
	// Add a verb to most recent route in the message
	public Message to(Verb verb);

	/**
	 * This method is called after a message is deserialized and ready to be processed by the target actor.
	 * The default implementation does nothing.
	 * This method can be overridden in a Message subclass to provide fine-grained tracing or additional message validation.
	 * <p>This method executes in the Actor's thread.</p>
	 * @param config
	 * @param actor The Actor that is about to process this message
	 * @return true if the message should be processed 
	 */
	public boolean preprocess(Config config, Actor actor);

	/**
	 * This method is called when a message is sent by an Actor. This happens just before the message is serialized for transport to its 
	 * current destination.
	 * A subclass of a default message can use this method to evaluate the message as well as the Actor that created or forwarded the message.
	 * This method can be overridden in a Message subclass to provide fine-grained tracing.
	 * <p>This method executes in the Actor's thread.</p>
	 * @param config
	 * @param actor The Actor that just processed this message
	 * @return true if the message should be processed 
	 */
	public boolean postprocess(Config config, Actor actor);

	/**
	 * This method is called when a message is sent to an Actor but the Actor has not overriden the processMessage method.
	 * The default behavior is to return an empty Stage. An alternate would be to put the inbound message back into the Stage (it will be 
	 * sent to the next item in the Route stack, if any).
	 * A subclass of a default message can use this method to provide alternate behavior or to forwarded the message.
	 * <p>This method executes in the Actor's thread.</p>
	 * @param config
	 * @param actor The Actor that received (but didn't process) this message
	 * @return The Stage to be processed
	 */
	public Stage notImplemented(Config config, Actor actor);
	
	/**
	 * This method is called when a message is sent by an Actor and the message as no further routes this it will be destroyed just after this method is called. 
	 * A subclass of a default message can use this method to evaluate the message as well as the Actor that created or forwarded the message to ensure
	 * that its demise is justified.
	 * <p>This method executes in the Actor's thread.</p>
	 * @param config
	 * @param actor The Actor that just processed this message
	 */
	public void terminal(Config config, Actor actor);
	
	/**
	 * Return a copy of the payload
	 * @return A new payload object
	 */
	public Map<String, Object> getPayload();

	/**
	 * Put new entry in payload
	 * @param value The Payload value
	 * @return Message builder-style
	 */
	public Message put(String key, Object value);
	
	/**
	 * Put new entry in payload convenience for a String
	 * @param key A string identifying the payload entry
	 * @param value The Payload value
	 * @return Message builder-style
	 */
	public Message put(String key, String value);

	public Message put(String key, Integer value);
	/**
	 * Put new entry in payload convenience for an ActorId
	 * @param key A string identifying the payload entry
	 * @param value The ActorId to add
	 * @return Message builder-style
	 */
	public Message put(String key, ActorId value);
	/**
	 * Find the info with the matching name (case insensitive).
	 * @param key
	 * @return
	 */
	public Object get(String key);
	
	public <T> T get(String key, Class<T> clazz);

	/**
	 * Find the Info record the matching name (case insensitive) and return its String Contents
	 * @param key
	 * @return
	 */
	public String getString(String key);
	/**
	 * Find the Info record the matching name (case insensitive) and return its String Contents
	 * @param key
	 * @return
	 */
	public ActorId getActorId(String key);
	
	/**
	 * Find the integer Info matching the name (case insensitive) and return its Integer Contents
	 * @param key
	 * @return
	 */
	public int getInteger(String key);

	/**
	 *  Remove entry from payload
	 * @param key The key to remove
	 * @return the message object
	 */
	public Message remove(String key);


}
