package net.sodacan.core;

import java.io.PrintStream;
import java.time.Clock;
import java.util.Map;
import java.util.Random;

import net.sodacan.core.config.ActorMetadata;
import net.sodacan.core.serialize.SerializerFactory;
import net.sodacan.core.persist.PersisterFactory;

/**
 * <p>The Sodacan configuration contains settings for specifying factories, singleton objects,
 * and static and dynamic settings.</p>
 * <p>Most configuration settings can be specified as functions. If the setting is "dynamic", 
 * the function will be called each time a value is requested. If the setting is "static",
 * the function is called once, during initialization and the result is saved and returned as
 * needed.
 * </p>
 * <p>The configuration
 * object is effectively read-only and is not thread-safe. Therefore, any of the functions
 * should be free of side-effects. In cases where side effects are required, this
 * configuration only returns a factory. In other words, the function returns a factory
 * which is used to create the target object and the factory's thread safety must be 
 * secured by the framework
 * </p>
 * <p>An exception to the thread safety rule is for global dynamic settings of which 
 * there are currently none.
 * </p>
 */
public interface Config {
	
	/**
	 * <p>Each host in the network has a unique number. This number is used to coordinate among multiple hosts in a network.
	 * The default is 1. Once assigned, a host number should be permanent. No duplicates.</p>
	 * @return An integer containing the host number
	 */
	public int getHostNumber();
	/**
	 * <p>The number of actorGroups in a network is more or less fixed. Changing this number is difficult. The number must be
	 * consistent on all hosts in the cluster. As such, this setting is maintained by the coordinator.
	 * </p>
	 * @return The number of actorGroups
	 */
	public int getActorGroups();

	public int getActorGroupReplicas();

	/**
	 * <p>The number of messages inFlight before backpressure is applied. This limit can change
	 * at any time.</p>
	 * @return
	 */
	public int getBackpressureLimit();

	public int getEviction();

	public ActorGroupAssigner getActorGroupAssigner();

	public MessageId createMessageId();
		
	/**
	 * Local static
	 * Return the current number of threads 
	 * @return
	 */
	public int getActorGroupThreads();

	public Coordinator getCoordinator();

	/**
	 * <p>Return the singleton host for this application instance</p>
	 * @return
	 */
	public Host getHost();

	/**
	 * <p>Return the clock used throuhout the system. Usually the default is adequate except for testing.</p>
	 * @return The Clock
	 */
	public Clock getClock();
	
	/**
	 * Get a Random Number Generator. The default implementation uses ThreadLocalRandom which is bound to the local Thread.
	 * Therefore, it is best to only use the result of this call on the stack. Or, at least don't use the returned generator 
	 * from more than one thread.
	 * @return an instance of Random
	 */
	public Random getRandom();

	/**
	 * Get the root directory for this host.
	 * Local/Status
	 * @return
	 */
	public String getRootDirectory();
	
	/**
	 * Create a actorGroup. 
	 * @return
	 */
	public ActorGroup createActorGroup(int actorGroupNumber);
	
//%%% ADD SENDER ACTORID For special actor.	And register sender.
//%%% listener actor (forwards directly to host without deserialization)
//%%% sender actorS
//%%% 
	/**
	 * <p>Create a new scheduler. The function is a factor for creating schedulers.
	 * </p>
	 * @return The new Scheduler subclass
	 */
	public Scheduler createScheduler(ActorGroup actorGroup);

	public ActorId createActorId(String actorType);

	public int getBackpressureWaitMs();

	public int getShutdownWaitMs();

	public String getProperty(String name);
	
	/**
	 * Static. Get the class associated with an ActorType which is either a string assigned to the actor
	 * by the ActorType annotation or the full class name.
	 * @param actorType string
	 * @return Either null or the Actor class, suitable to use to instantiate that actor.
	 */
	public Class<? extends Actor> getActorClass(String actorType);
	
	/**
	 * Static. We need to keep ActorMetadata along with the ActorClass.
	 */
	public ActorMetadata getActorMetadata(String actorType);

	/**
	 * The SerializerFactory is instantiated during configuration
	 * @return
	 */
	public SerializerFactory getSerializerFactory();

	/**
	 * The PersisterFactory provides access to persistent services for an Actor.
	 * The Factory is instantiated during configuration. The factory implementation
	 * can decide during construction how it will handle requests for multiple Actor threads.
	 * @return PersisterFactory implementation.
	 */
	public PersisterFactory getPersisterFactory();
	/**
	 * Return all of the actorClasses that we know of
	 * @return
	 */
	public Map<String, Class<? extends Actor>> getActorTypes();

	public Actor createActor(ActorId actorId);

	public Message createMessage();
	
	/**
	 * Send a list of actorType names and actor class name pairs
	 * @param out
	 */
	public void printConfig(PrintStream out);

	/**
	 * Send a list of actorType names and actor class name pairs
	 * @param out
	 */
	public void listActors(PrintStream out);

}
