package net.sodacan.core.scheduler;

import java.io.Closeable;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;

import net.sodacan.core.Actor;
import net.sodacan.core.ActorGroup;
import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Message;
import net.sodacan.core.Scheduler;
import net.sodacan.core.SodacanException;
import net.sodacan.core.Stage;
import net.sodacan.core.actor.ActorEntry;
import net.sodacan.core.message.Evict;

public class DefaultScheduler implements Scheduler, Closeable {

	private Config config;
	protected ActorGroup actorGroup;
	// Statistics
	private AtomicInteger messageCount = new AtomicInteger(0);
	private AtomicInteger messageLoad = new AtomicInteger(0);
	private AtomicInteger maxMessageLoad = new AtomicInteger(0);
	private AtomicInteger evictionCount = new AtomicInteger(0);
	private AtomicInteger totalSleepTime = new AtomicInteger(0);
	private AtomicInteger maxThreadQueueDepth = new AtomicInteger(0);

	private final ReentrantLock newMessageLock = new ReentrantLock();
	
	private ThreadPoolExecutor pool;

	private Map<ActorId, ActorEntry> actorEntries = new ConcurrentHashMap<>();

	/**
	 * Construct a default scheduler, one per ActorGroup
	 * @param config Configuration used by this scheduler
	 */
	@SuppressWarnings("serial")
	public DefaultScheduler(Config config) {
		this.config = config;
		// Thread pool
		pool = (ThreadPoolExecutor) Executors.newCachedThreadPool();
	}

	public ActorGroup getActorGroup() {
		return actorGroup;
	}

	public void setActorGroup(ActorGroup actorGroup) {
		this.actorGroup = actorGroup;
	}

	/**
	 * Add a message to the work queue
	 */
	@Override
	public boolean addMessage(Message message) {
		return addMessage(message, false);
	}

	/**
	 * <p>There are no queue limits in the system so the best way to 
	 * regulate flow is to check how many in-flight messages
	 * are in-process and sleep for a while until the queue settles down.</p>
	 * @return
	 */
	protected boolean backpressureWait() {
		while (getMessageLoad()>= config.getBackpressureLimit()) {
			// oops, no room at the inn
//			System.out.print("waiting...");
			try {
				Thread.sleep(config.getBackpressureWaitMs());
				incrementSleepTime();
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
//			int load = getMessageLoad();
//			if (load>0) {
//				System.out.println(" OK, load is now at=" + load);
//			}
		}
		return true;
	}
	/**
	 * <p>Evict a anActorEntry. Send an Evict message to get the job done. </p>
	 * @param actorEntry
	 */
	protected void evict(ActorEntry actorEntry) {
		actorEntry.queueMessage(new Evict(actorEntry.getActorId()));
		actorEntries.remove(actorEntry.getActorId());
		// %%% This isn't right, we could accept a new message before the
		// eviction is complete
	}

	/**
	 * This method operates inside the synchronized addMessage method
	 * to make room, if needed, in the collection of ActorEntries for a new ActorEntry.
	 */
	protected void makeRoomInActorEntries() {
		int eviction = config.getEviction();
		int aeSize = actorEntries.size();
		// Is it time, yet?
		if (aeSize<(config.getActorGroupThreads()+eviction)) {
			return;	// No
		}
		// Make an array of all current actor entries
		ActorEntry[] ae = actorEntries.values().toArray(new ActorEntry[aeSize]);
		// Sort by usage, putting the least used at the front
		Arrays.sort(ae, Comparator.comparingLong(ActorEntry::getUsage));
		// Evict and remove the first N entries
		for (int x=0;x <eviction; x++) {
			evict(ae[x]);
		}
	}

	protected synchronized ActorEntry addActorEntry(ActorId actorId) {
		ActorEntry actorEntry = null;
		try {
//			newMessageLock.lock();
			makeRoomInActorEntries();
			actorEntry = new ActorEntry(config, this, actorId);
			actorEntries.put(actorId, actorEntry);
			pool.execute(actorEntry);
		} catch (Exception e1) {
			// TODO Auto-generated catch block
			e1.printStackTrace();
		} finally {
//			newMessageLock.unlock();			
		}
		return actorEntry;		
	}
	/**
	 * <p>Add a message with backpressure; This method will block if too many messages
	 * are already queued but not yet processed.</p>
	 * <p>After backpressure is considered, queue up the message. This part of the 
	 * process has to be synchronized.</p>
	 * <p>First, if we don't have an actor entry, create it now.
	 * Then, queue the message. Finally, submit the ActorEntry (Runnable) to the executor.
	 * </p>
	 * 
	 * @param message The message to be processed
	 * @param force If true, the message is added to the queue without considering backpressure
	 */
	@Override
	public boolean addMessage(Message message, boolean force) {
		if (!force) {
			backpressureWait();
		}

		ActorId targetActorId = null;
		targetActorId = message.getTargetActorId();
		if (targetActorId==null) {
			throw new RuntimeException("In scheduler, Message has no targetActorId");
		}
		ActorEntry actorEntry = actorEntries.get(targetActorId);
		// If no entry yet, create an actorEntry. The actor is created later.
		// We start up a thread to empty the queue.
		if (actorEntry==null) {
			actorEntry = addActorEntry(targetActorId);
		}
		// All good to submit the message now
		actorEntry.queueMessage(message);
		// Keep track of maximum queue depth
		int depth = actorEntry.getQueueSize();
		if (depth > maxThreadQueueDepth.get()) {
			maxThreadQueueDepth.set(depth);
		}
		return true;			
	}
	/**
	 * Debugging aid
	 */
	public void auditActorEntries() {
//		newMessageLock.lock();
		try {
			for (Map.Entry<ActorId, ActorEntry> entry : actorEntries.entrySet()) {
				int cnt = entry.getValue().getQueueSize();
				if (cnt>0) {
					System.out.println( entry.getKey() + " queue size=" + cnt);
					pool.execute(entry.getValue());
				}
//				if (entry.getValue().running.get()) {
//					System.out.println("Running: " + entry.getKey());
//				}
			}
		} finally {
//			newMessageLock.unlock();
		}
	}
	
	@Override
	public void evictAll() {
//		newMessageLock.lock();
		try {
			// Make an array of all current actor entries
			int aeSize = actorEntries.size();
			ActorEntry[] ae = actorEntries.values().toArray(new ActorEntry[aeSize]);
			for (int x=0;x <ae.length; x++) {
				evict(ae[x]);
			}
		} finally {
//			newMessageLock.unlock();
		}
	}
	
	@Override
	public void waitForMessagesToFinish() {
		try {
		// When message load goes to zero, shutdown is done
		while (0!=getMessageLoad()) {
			System.out.println("Shutdown: Waiting for " + getMessageLoad() + " messages to finish..." );
				Thread.sleep(config.getShutdownWaitMs());
//debug				if (getMessageLoad()<50) {
//					auditActorEntries();
//				}
		}
		} catch (InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	@Override
	public Config getConfig() {
		return config;
	}

	@Override
	public void increaseMessageLoad() {
		int load = messageLoad.incrementAndGet();
		if (load>maxMessageLoad.get()) {
			maxMessageLoad.set(load);
		}
	}

	@Override
	public void reduceMessageLoad() {
		messageLoad.decrementAndGet();		
	}

	@Override
	public int getMessageLoad() {
		return messageLoad.get();
	}
	
	@Override
	public int getMaxMessageLoad() {
		return maxMessageLoad.get();
	}

	@Override
	public void incrementSleepTime() {
		totalSleepTime.addAndGet(getConfig().getBackpressureWaitMs());		
	}

	@Override
	public void increaseMessageCount() {
		messageCount.incrementAndGet();
	}

	@Override
	public int getMessageCount() {
		return messageCount.get();
	}

	@Override
	public int getSleepTime() {
		return totalSleepTime.get();
	}

	@Override
	public int getActorCount() {
		return actorEntries.size();
	}

	@Override
	public int getMaxThreadQueueDepth() {
		return maxThreadQueueDepth.get();
	}
	@Override
	public void increaseEvictionCount() {
		evictionCount.incrementAndGet();		
	}
	
	@Override
	public int getEvictionCount() {
		return evictionCount.get();
	}

	@Override
	public int getThreadPoolSize() {
		return pool.getActiveCount();
	}
	
	@Override
	public void close() {
		try {
			evictAll();
			waitForMessagesToFinish();
			pool.awaitTermination(1, TimeUnit.SECONDS);
			pool.shutdown();
			System.out.println("Schduler orderly Shutdown complete for " + getActorGroup());
		} catch (Exception e) {
			throw new RuntimeException("Error closing Scheduler for ActorGroup ",e);
		}
	}


}
