package net.sodacan.core.message;

import java.util.LinkedList;

import net.sodacan.core.ActorId;

/**
 * Event messages generally represent a business event which results
 * in a state-change in the target actor. Such messages can have complex routing 
 * behavior. For example, an event might be enhanced by other actors before being
 * routed on to the final destination.
 */
public abstract class EventMessage extends AbstractMessage {
	// Message routes are in a stack: The top of the stack is the current route. 
	private LinkedList<Route> routes = new LinkedList<>();

	public EventMessage(ActorId target) {
		super(target);
	}
	
	/**
	 * Get the target by peeking at the top-of-stack of the routing table.
	 */
	@Override
	public ActorId getTargetActorId() {
		Route route = peekRoute();
		if (route==null) {
			throw new RuntimeException("Event message has no route's left");
		}
		return route.getTarget();
	}

	/**
	 * A message route carries the information needed to get a message to where 
	 * it is going and what it's purpose is when it gets there.
	 * Routes in a message are organized into a stack to facilitate complex
	 * message flows. The top of the stack describes where the message is currently heading.
	 * The bottom of the stack is often the Actor that initiated the message in the first place or perhaps
	 * a different reply-to Actor.
	 */
	public static class Route {
		private String verb;
//		private ActorId sourceActor;
		private ActorId target;
		
		public Route(String verb, ActorId target) {
			this.verb = verb;
			this.target = target;
		}
		public String getVerb() {
			return verb;
		}
		public ActorId getTarget() {
			return target;
		}
	}
	
	public Route peekRoute() {
		Route route = routes.peek();
		return route;
	}

	public Route popRoute() {
		Route route = routes.pop();
		return route;
	}
	
	public void pushRoute(Route route) {
		routes.push(route);
	}
}

