package net.sodacan.core.config;

import java.io.IOException;
import java.io.PrintStream;
import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.BiFunction;
import java.util.function.Function;

import net.sodacan.core.Actor;
import net.sodacan.core.ActorGroup;
import net.sodacan.core.ActorGroupAssigner;
import net.sodacan.core.ActorId;
import net.sodacan.core.ActorIdFactory;
import net.sodacan.core.Config;
import net.sodacan.core.Coordinator;
import net.sodacan.core.Host;
import net.sodacan.core.Scheduler;
import net.sodacan.core.actor.DefaultActorIdFactory;
import net.sodacan.core.actorgroup.DefaultActorGroup;
import net.sodacan.core.actorgroup.DefaultActorGroupAssigner;
import net.sodacan.core.coordinator.SingleHostCoordinator;
import net.sodacan.core.host.DefaultHost;
import net.sodacan.core.scheduler.DefaultScheduler;
import net.sodacan.core.util.ClassUtilities;
/**
 * Default implementation of the Config interface.
 * 
 * @see net.sodacan.core.config.DefaultConfig.Builder
 */
public class DefaultConfig implements Config {
		// local static
		private int hostNumber;

		// global static
		private int actorGroups;

		// global static
		private int actorGroupReplicas;

		// local dynamic
		private Function<Config,Integer> backpressureLimitFn;
		// local dynamic
		private Function<Config,Integer> backpressureWaitMsFn;
		// local dynamic	
		private Function<Config,Integer> evictionFn;
		// local dynamic	
		private Function<Config,Integer> actorGroupThreadsFn;
		// local dynamic	
		private Function<Config,Integer> shutdownWaitMsFn;

		// local singleton  ***** Should be a Factory per actor????? for thread safety
		private ActorIdFactory actorIdFactory;
		// local static class
		private Host host;
		// local singleton
		private ActorGroupAssigner actorGroupAssigner;
		// local singleton
		private Coordinator coordinator;
		// local factory
		private Function<Config,Scheduler> schedulerFn;
		// local factory
		private BiFunction<Config,Integer,ActorGroup> actorGroupFn;

		private Map<String,Class<? extends Actor>> actorTypes;
		private Map<String,BiFunction<Config,ActorId,Actor>> actorFactories;

		/**
		 * <p>Construct a configuration object. Many of the settings can be specified as
		 * functions. This allows some behavior in configuring the system. For example,
		 * the message "eviction" setting could change depending on time of day.</p>
		 * <p>In a clustered configuration, some of the configuration settings 
		 * are shared among all hosts. In these cases, the parameter is 
		 * described as "dynamic". On the other hand, some configuration settings 
		 * are unique to a specific host. These settings are described as local.
		 * In a single host configuration, there is no difference between local 
		 * and global settings.</p>
		 * @param builder The builder object used to construct this configuration
		 */
		private DefaultConfig(DefaultConfig.Builder builder) {
			// Static, we don't need the function past this point
			hostNumber = builder.hostNumberFn.apply(this);
			if (hostNumber<0) {
				throw new RuntimeException("The hostNumber must be positive");
			}
			actorGroupReplicas = builder.actorGroupReplicasFn.apply(this);
			if (actorGroupReplicas < 0) {
				throw new RuntimeException("The number of actorGroup replicas must be a positive number");			
			}
			// Static, we don't need the function past this point
			actorGroups = builder.actorGroupsFn.apply(this);
			if (actorGroups<1) {
				throw new RuntimeException("The number of actorGroups must be 1 or greater");
			}
			// Static, we don't need the function past this point
			coordinator = builder.coordinatorFn.apply(this);
			// Static, we don't need the function past this point
			host = builder.hostFn.apply(this);
			// Static/dynamic The number of threads is only used when constructing
			// a actorGroup, it won't change until that actorGroup is restarted.
			actorGroupThreadsFn = builder.actorGroupThreadsFn;

			// Static, we don't need the function past this point
			actorGroupAssigner = builder.actorGroupAssignerFn.apply(this);
			// Static, we don't need the function past this point
			actorIdFactory = builder.actorIdFactoryFn.apply(this);
			// Singleton Factory
			schedulerFn = builder.schedulerFn;
			// Singleton Factory
			actorGroupFn = builder.actorGroupFn;
			// Local Static
			this.actorTypes = builder.actorTypes;
			// Dynamic
			this.backpressureLimitFn = builder.backpressureLimitFn;
			// Dynamic
			this.evictionFn = builder.evictionFn;
			// Dynamic
			this.backpressureWaitMsFn = builder.backpressureWaitMsFn;
			// Dynamic
			this.shutdownWaitMsFn = builder.shutdownWaitMsFn;
			// Factory
			this.actorFactories = builder.actorFactories;
			// Create a default actor factory for those not explicitly configured
			addMissingActorFactories();
		}

		/**
		 * <p>
		 * If any actor factories are not specified, create factories that
		 * use the default actor signature of <code>Actor(Config,ActorId);</code>
		 * </p>
		 */
		protected void addMissingActorFactories() {
			try {
				for (Entry<String,Class<? extends Actor>> entry : actorTypes.entrySet()) {
					// If factory already defined, then we're done with this one
					if (actorFactories.containsKey(entry.getKey())) {
						continue;
					}
					// Need to make a standard factory function.
					Class<? extends Actor> clazz = entry.getValue();
					// Find the constructor for the target actor
					Constructor<? extends Actor> ctor = clazz.getConstructor(Config.class, ActorId.class);
					if (ctor==null) {
						throw new RuntimeException("Actor " + entry.getKey() + " Does not support a standard constructor, use a custom factory");
					}
					// Create the function
					BiFunction<Config,ActorId,Actor> f = (config,actorId) -> {
						try {
							return (Actor) ctor.newInstance(config,actorId);
						} catch (Exception e) {
							throw new RuntimeException("Error constructing actor " + entry.getKey(), e);
						}
					};
					actorFactories.put(entry.getKey(),f);
				}
			} catch (Exception e) {
				throw new RuntimeException("Error creating default actor factories", e);
			}
		}
		public static class Builder {
			// local static
			private Function<Config,Integer> hostNumberFn = config -> 1;
			// global static
			private Function<Config,Integer> actorGroupsFn = config -> 5;
			private Function<Config,Integer> actorGroupReplicasFn = config -> 0;
			private Function<Config,Integer> backpressureLimitFn = config -> 100000;
			private Function<Config,Integer> backpressureWaitMsFn = config -> 200;
			private Function<Config,Integer> shutdownWaitMsFn = config -> 200;
			private Function<Config,Integer> evictionFn = config -> 8;
			private Function<Config,Integer> actorGroupThreadsFn = config -> 20;		
			private Function<Config,ActorGroupAssigner> actorGroupAssignerFn = config -> new DefaultActorGroupAssigner(config);
			private Function<Config,Scheduler> schedulerFn = config -> new DefaultScheduler(config);
			private BiFunction<Config,Integer,ActorGroup> actorGroupFn = (config,id) -> new DefaultActorGroup(config,id);
			private Function<Config,Coordinator> coordinatorFn = config -> new SingleHostCoordinator(config);
			private Function<Config,Host> hostFn = config -> new DefaultHost(config);		
			private Function<Config,ActorIdFactory> actorIdFactoryFn = config -> new DefaultActorIdFactory();

			private Map<String,Class<? extends Actor>> actorTypes = new HashMap<>();
			private Map<String,BiFunction<Config,ActorId,Actor>> actorFactories = new
					HashMap<>();

			/**
			 * Local Static
			 * @param hostNumber
			 * @return
			 */
			public Builder hostNumber(int hostNumber) {
				this.hostNumberFn = config -> hostNumber;
				return this;
			}

			public Builder hostNumber(Function<Config,Integer> hostNumberFn) {
				this.hostNumberFn = hostNumberFn;
				return this;
			}

			/**
			 * Specify the number of actorGroups in the system. Default is 1
			 * Global Static
			 * @param actorGroups
			 * @return
			 */
			public Builder actorGroups(int actorGroups) {
				actorGroupsFn = config -> actorGroups;
				return this;
			}

			public Builder actorGroups(Function<Config,Integer> actorGroupsFn) {
				this.actorGroupsFn = actorGroupsFn;
				return this;
			}
			/**
			 * <p>The number of replicas of a actorGroup to maintain. Zero is valid. 
			 * It means that only a primary actorGroup exists. The default is zero.
			 * </p>
			 * @param actorGroupReplicas
			 * @return
			 */
			public Builder actorGroupReplicas(int actorGroupReplicas) {
				actorGroupReplicasFn = config -> actorGroupReplicas;
				return this;
			}

			public Builder actorGroupReplicas(Function<Config,Integer> actorGroupReplicasFn) {
				this.actorGroupReplicasFn = actorGroupReplicasFn;
				return this;
			}

			/**
			 * Global dynamic
			 * <p>This is the extra number of threads we add to allow that number of ActorEntry threads
			 * to be removed to make room for new messages. If this value is zero, it's very likely to
			 * cause a deadlock in the ActorGroup because there is more work to do but no Actors can be
			 * evicted in order to perform that work.
			 * </p>
			 * <p>This number should be above 5 to improve performance. The less the number, 
			 * the more often the eviction process has to begin.</p>
			 * @param eviction
			 * @return
			 */
			public Builder eviction(int eviction) {
				this.evictionFn = config -> eviction;
				return this;
			}

			public Builder eviction(Function<Config,Integer> evictionFn) {
				this.evictionFn = evictionFn;
				return this;
			}

			/**
			 * Local dynamic (maybe) Change effective in new actorGroups only
			 * @param actorGroupThreads
			 * @return
			 */
			public Builder actorGroupThreads(int actorGroupThreads) {
				this.actorGroupThreadsFn = config -> actorGroupThreads;
				return this;
			}
			
			public Builder actorGroupThreads(Function<Config,Integer> actorGroupThreadsFn) {
				this.actorGroupThreadsFn = actorGroupThreadsFn;
				return this;
			}

			/**
			 * Global dynamic
			 * @param backpressureLimitFn
			 * @return
			 */
			public Builder backpressureLimit(Function<Config,Integer> backpressureLimitFn) {
				this.backpressureLimitFn = backpressureLimitFn;
				return this;
			}

			public Builder backpressureLimit(int backpressureLimit) {
				this.backpressureLimitFn = config -> backpressureLimit;
				return this;
			}

			/**
			 * Specify the backpressure limit in an ActorGroup. When the number of in-flight messages
			 * exceeds this limit, inbound messages are blocked.
			 * Global dynamic
			 * @param backpressureWaitMsFn
			 * @return
			 */
			public Builder backpressureWaitMs(Function<Config,Integer> backpressureWaitMsFn) {
				this.backpressureWaitMsFn = backpressureWaitMsFn;
				return this;
			}

			/**
			 * Specify the backpressure limit in an ActorGroup. When the number of in-flight messages
			 * exceeds this limit, inbound messages are blocked.
			 * Global static
			 * @param backpressureWaitMs
			 * @return
			 */
			public Builder backpressureWaitMs(int backpressureWaitMs) {
				this.backpressureWaitMsFn = config -> backpressureWaitMs;
				return this;
			}

			/**
			 * Local singleton
			 * @param actorGroupAssignerFn
			 * @return
			 */
			public Builder actorGroupAssigner(Function<Config,ActorGroupAssigner> actorGroupAssignerFn) {
				this.actorGroupAssignerFn = actorGroupAssignerFn;
				return this;
			}
			
			/**
			 * 
			 * @param actorIdFactoryFn A function that will return the ActorIdFactory
			 * @return
			 */
			public Builder actorIdFactory(Function<Config,ActorIdFactory> actorIdFactoryFn) {
				this.actorIdFactoryFn = actorIdFactoryFn;
				return this;
			}

			/**
			 * global static
			 * @param packageName
			 * @return
			 */
			public Builder registerActorsInPackage(String packageName) {
				try {
					this.actorTypes.putAll(ClassUtilities.getActorClassesFromPackage(packageName));
				} catch (IOException e) {
					throw new RuntimeException("Trouble loading Actors from package " + packageName, e);
				}
				return this;
			}

			/**
			 * global static
			 * Explicitly register an actor type name to actor class. If the actor
			 * has an ActorType annotation, this registration is not needed. If this is called after
			 * the package containing the actor is registered, then this setting takes precidence.
			 * @param actorType
			 * @param type
			 * @return Builder object to allow chaining
			 */
			public Builder registerActorType(String actorType, Class<? extends Actor> type) {
				this.actorTypes.put(actorType, type);
				return this;
			}

			/**
			 * global static
			 * Specify a custom actor factory. Is an actor that has a constructor 
			 * taking Config and ActorId, a custom actor is not required, but can be specified. 
			 * @param actorType
			 * @param actorFactory
			 * @return
			 */
			public Builder actorFactory(String actorType, BiFunction<Config,ActorId,Actor> actorFactory) {
				this.actorFactories.put(actorType, actorFactory);
				return this;
			}
			
			/**
			 * global dynamic
			 * @param ms
			 * @return
			 */
			public Builder shutdownWaitMs(Function<Config,Integer> ms) {
				this.shutdownWaitMsFn = ms;
				return this;
			}

			public Builder shutdownWaitMs(int ms) {
				this.shutdownWaitMsFn = config -> ms;
				return this;
			}

			/**
			 * Local singleton
			 * @param coordinatorFn
			 * @return
			 */
			public Builder coordinator(Function<Config,Coordinator> coordinatorFn) {
				this.coordinatorFn = coordinatorFn;
				return this;
			}
			
			/**
			 * local singleton
			 * @param hostFn
			 * @return
			 */
			public Builder host(Function<Config,Host> hostFn) {
				this.hostFn = hostFn;
				return this;
			}

			/**
			 * Local factory
			 * @param schedulerFn
			 * @return
			 */
			public Builder schedulerCreator(Function<Config,Scheduler> schedulerFn) {
				this.schedulerFn = schedulerFn;
				return this;
			}
			/**
			 * Local factory
			 * 
			 * @param actorGroupFn
			 * @return
			 */
			public Builder actorGroupCreator(BiFunction<Config,Integer,ActorGroup> actorGroupFn) {
				this.actorGroupFn = actorGroupFn;
				return this;		
			}
			/**
			 * Set a system property
			 * @param name
			 * @param value
			 * @return
			 */
			public Builder property(String name, String value) {
				System.setProperty(name,value);
				return this;
				
			}
			
			/**
			 * Build the Config object
			 * @return
			 */
			public Config build() {
				return new DefaultConfig(this);
			}
		}
		/**
		 * <p>Each host in the network has a unique number. This number is used to coordinate among multiple hosts in a network.
		 * The default is 1. Once assigned, a host number should be permanent. No duplicates.</p>
		 * @return An integer containing the host number
		 */
		@Override
		public int getHostNumber() {
			return hostNumber;
		}
		/**
		 * <p>The number of actorGroups in a network is more or less fixed. Changing this number is difficult. The number must be
		 * consistent on all hosts in the cluster. As such, this setting is maintained by the coordinator.
		 * </p>
		 * @return The number of actorGroups
		 */
		@Override
		public int getActorGroups() {
			return actorGroups;
		}

		@Override
		public int getActorGroupReplicas() {
			return actorGroupReplicas;
		}

		/**
		 * <p>The number of messages inFlight before backpressure is applied. This limit can change
		 * at any time.</p>
		 * @return
		 */
		@Override
		public int getBackpressureLimit() {
			int limit = backpressureLimitFn.apply(this);
			if (limit<1) {
				throw new RuntimeException("Backpressure limit must be a positive number");
			}
			return limit;
		}

		@Override
		public int getEviction() {
			return evictionFn.apply(this);
		}

		@Override
		public ActorGroupAssigner getActorGroupAssigner() {
			return actorGroupAssigner;
		}

		@Override
		public ActorIdFactory getActorIdFactory() {
			return actorIdFactory;
		}
		
		/**
		 * Local static
		 * Return the current number of threads 
		 * @return
		 */
		@Override
		public int getActorGroupThreads() {
			return actorGroupThreadsFn.apply(this);
		}

		@Override
		public Coordinator getCoordinator() {
			return coordinator;
		}

		/**
		 * <p>Return the singleton host for this application instance</p>
		 * @return
		 */
		@Override
		public Host getHost() {
			return host;
		}
		
		/**
		 * Create a actorGroup. 
		 * @return
		 */
		@Override
		public ActorGroup createActorGroup(int actorGroupNumber) {
			return actorGroupFn.apply(this,actorGroupNumber);
		}
	//%%% ADD SENDER ACTORID For special actor.	And register sender.
	//%%% listener actor (forwards directly to host without deserialization)
	//%%% sender actorS
	//%%% 
		/**
		 * <p>Create a new scheduler. The function is a factor for creating schedulers.
		 * </p>
		 * @return The new Scheduler subclass
		 */
		@Override
		public Scheduler createScheduler() {
			return schedulerFn.apply(this);
		}

		@Override
		public int getBackpressureWaitMs() {
			return backpressureWaitMsFn.apply(this);
		}

		@Override
		public int getShutdownWaitMs() {
			return shutdownWaitMsFn.apply(this);
		}

		@Override
		public String getProperty(String name) {
			return System.getProperty(name);
		}
		
		/**
		 * Get the class associated with an ActorType which is either a string assigned to the actor
		 * by the ActorType annotation or the full class name.
		 * @param actorType string
		 * @return Either null or the Actor class, suitable to use to instantiate that actor.
		 */
		@Override
		public Class<? extends Actor> getActorClass(String actorType) {
			return actorTypes.get(actorType);
		}

		@Override
		public Actor createActor(ActorId actorId) {
			String actorType = actorId.getType();
			if (actorType==null) {
				throw new RuntimeException("Null ActorType not allowed");
			}
			BiFunction<Config,ActorId,Actor> factory = actorFactories.get(actorType);
			if (factory==null) {
				throw new RuntimeException("No factory for actorType " + actorType);
			}
			return factory.apply(this, actorId);
		}
		
		/**
		 * Send a list of actorType names and actor class name pairs
		 * @param out
		 */
		@Override
		public void printConfig(PrintStream out) {
			out.println("Local Static");
			out.println(" actorIdFactory=" + getActorIdFactory());
			out.println(" Host: " + getHost());
			out.println(" hostNumber=" + getHostNumber());
			out.println("Global Static");
			out.println("  actorGroups=" + getActorGroups());
			out.println("  actorGroupReplicas=" + getActorGroupReplicas());
			out.println("Local Static factory");
			out.println(" Coordinator: " + getCoordinator());
			out.println(" actorGroupAssigner=" + getActorGroupAssigner());
			out.println(" ActorIdFactory=" + getActorIdFactory());
			out.println("Global Dynamic");
			out.println(" backpressureLimit=" + getBackpressureLimit());
			out.println(" backpressureWaitMs=" + getBackpressureWaitMs());
			out.println(" eviction=" + getEviction());
			out.println(" actorGroupThreads=" + getActorGroupThreads());
			out.println(" shutdownWaitMs=" + getShutdownWaitMs());
			out.println("Property: os.name=" + getProperty("os.name"));
			out.println("actors: ");
			listActors(out);
		}
		
		/**
		 * Send a list of actorType names and actor class name pairs
		 * @param out
		 */
		@Override
		public void listActors(PrintStream out) {
			for (Entry<String,Class<? extends Actor>> entry : actorTypes.entrySet()) {
				out.print("  ");
				out.print(entry.getKey());
				out.print("=");
				out.println(entry.getValue().getName());
			}
		}


}
