package net.sodacan.core.actorgroup;


import java.io.IOException;

import net.sodacan.core.ActorGroup;
import net.sodacan.core.Config;
import net.sodacan.core.Host;
import net.sodacan.core.Message;
import net.sodacan.core.Scheduler;
import net.sodacan.core.Stage;

public abstract class AbstractActorGroup implements ActorGroup {
	private Config config;
	private int actorGroupNumber;
	private Scheduler scheduler;
	protected Host host = null;
	public AbstractActorGroup( Config config, int actorGroupNumber) {
		this.config = config;
		this.actorGroupNumber = actorGroupNumber;
		// Verify that actorGroup number is valid
		if (actorGroupNumber > config.getActorGroups() || actorGroupNumber < 0) {
			throw new RuntimeException("actorGroup number must be less than the number of actorGroups and not negative");
		}
		// Need a scheduler
		scheduler = config.createScheduler();
		scheduler.setActorGroup(this);
		System.out.println("Actor group created: " + this);
	}

	
	@Override
	public void setHost(Host host) {
		this.host = host;
	}


	@Override
	public Scheduler getScheduler() {
		return scheduler;
	}

	@Override
	public void close() throws IOException {
		scheduler.close();
		System.out.println("Actor group closed: " + this);		
	}

	public Config getConfig() {
		return config;
	}
	public int getActorGroupNumber() {
		return actorGroupNumber;
	}

	public int getMessageLoad() {
		return scheduler.getMessageLoad();
	}
	
	public void increaseMessageLoad() {
		scheduler.increaseMessageLoad();
	}

	public void reduceMessageLoad() {
		scheduler.reduceMessageLoad();
	}

	@Override
	public String toString() {
		return getClass().getSimpleName() + "=" + getActorGroupNumber();
	}

	/**
	 * <P>Queue this message for processing by schduler
	 * </p>
	 */
	@Override
	public void addMessage(Message message) {
		// Check the actorGroup for the target actor
		getScheduler().addMessage(message);
	}

	/**
	 * Propagate the message up to the host for routing
	 */
	@Override
	public void submit(Message message) {
		host.submit(message);
	}


	@Override
	public void submit(Stage stage) {
		host.submit(stage);
	}
	
}
