package net.sodacan.core.actor;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import net.sodacan.core.Actor;
import net.sodacan.core.ActorGroup;
import net.sodacan.core.ActorId;
import net.sodacan.core.Config;
import net.sodacan.core.Message;
import net.sodacan.core.Precheck;
import net.sodacan.core.Scheduler;
import net.sodacan.core.Stage;
import net.sodacan.core.message.Evict;

/**
 * <p>The scheduler for a actorGroup manages a map of ActorEntries. 
 * Each entry wraps a single actor instance. This entry
 * is what contains the "Runnable" entity for purposes of executing
 * the actor's precheck and processMessage methods as messages arrive.</p>
 * <p>Each entry maintains a queue of inbound messages which are fed 
 * to the actor one at a time. In this way, an actor instance is safely 
 * single-threaded. As long as an actor makes no attempt to go outside
 * of it's declare domain, it need not be concerned about thread safety.
 * </p>
 */	
public class ActorEntry implements Runnable {
//	private AtomicBoolean running = new AtomicBoolean(false);
	protected ActorId actorId;
	protected Actor actor = null;
	private BlockingQueue<Message> queue = new LinkedBlockingQueue<>();
	private List<Message> holding = new LinkedList<>();
	protected Scheduler scheduler;
	protected Config config;
	protected long usage = 0;

	public ActorId getActorId() {
		return actorId;
	}

	public long getUsage() {
		return usage;
	}

	/**
	 * <p>Construct a new ActorEntry and instantiate and actor contained within
	 * it using a configured actor factory. The only thing an actor "knows" after
	 * creation is its ActorId. Additional saved state will arrive shortly
	 * after creation, if needed. This is necessary to maintain the spirit of
	 * the actor model in Sodacan.</p>
	 * @param config 
	 * @param scheduler
	 * @param actorId
	 */
	public ActorEntry(Config config, Scheduler scheduler, ActorId actorId) {
		this.config = config;
		this.scheduler = scheduler;
		this.actorId = actorId;
	}
	
	/**
	 * Callable from any thread to add a message to this queue.
	 * Usually called from the scheduler.
	 * @param message
	 */
	public void queueMessage(Message message) {
		queue.add(message);
		scheduler.increaseMessageLoad();
		if (message instanceof Evict) {
			scheduler.increaseEvictionCount();				
		} else {
			scheduler.increaseMessageCount();
		}
	}
	/**
	 * Process the stage - we're still in the ActorEntry run thread.
	 * MUST BE DONE IN HOST in case ActorGroup has changed
	 * But for unit tests, we'll submit locally if ActorGroup
	 * is null.
	 * @param stage
	 */

	protected void handleStage(Stage stage) {
		ActorGroup actorGroup = scheduler.getActorGroup();
		if (actorGroup!=null) {
			actorGroup.submit(stage);
		} else {
			for (Message msg : stage.getMessages()) {
				scheduler.addMessage(msg, true);
			}
		}
	}
	
	protected void callProcessMessage(Message message ) {
		// Usage count needed by LRU
		usage++;
		// process message
		Stage stage = actor.processMessage(message);
		// Reduce the overall message load count, after message is processed
		scheduler.reduceMessageLoad();
		handleStage(stage);
	}
	
	/**
	 * Call the "request" method in the actor.
	 * Note: The original message is still in-flight, this may/will generate new messages.
	 * @param pca
	 * @param message
	 */
	protected void callRequest(Precheck pca, Message message) {
		// process message
		Stage stage = pca.request(message);
		// And the state
		handleStage(stage);
	}
	/**
	 * Return true if this is an eviction
	 * @param message
	 * @return
	 */
	protected boolean isEviction(Message message) {
		// If we're being evicted, just get out.
		if (message instanceof Evict) {
			scheduler.reduceMessageLoad();
			int queueSize = queue.size();
			if (queueSize>0) {
				System.out.println("Eviction with " + queueSize + " in queue");
			}
			return true;
		}
		return false;
	}
	
	@Override
	public void run() {
		try {
			// The Actor instance needs to exist now (initializing an actor could take time.
			// The time is charged to the actor, a good thing.)
			if (actor==null) {
				actor = config.createActor(actorId);
			}
			if (actor instanceof Precheck pca) {
				// An actor with precheck is more complicated
				while (true) {
					Message message = queue.take();
					if (isEviction(message)) {
						return;
					}
					// If this message needs checking, get ready to set it aside
					// If the message already passes, then just let it through
					// like a regular message
					if (pca.hold(message) && !pca.release(message)) {
						// OK, we are holding it, so initiate whatever request(s) are 
						// needed that will satisfy the conditions of hold
						callRequest(pca, message);
						// And add the message to our hold list
						holding.add(message);
						// And don't process the message, yet
						continue;
					}
					// Message not being held
					callProcessMessage(message);
					// Look at each message in holding to see if we can release anything
					for (Message m : holding) {
						if (pca.release(message)) {
							holding.remove(m);
							callProcessMessage(message);
						}
					}
				}
			} else {
				// Normal actor without precheck
				while (true) {
					Message message = queue.take();
					if (isEviction(message)) {
						return;
					}
					callProcessMessage(message);
				}
			}
		} catch (Exception e) {
			throw new RuntimeException("Exception in actor " + actorId, e);
		}
	}
	
	public int getQueueSize() {
		return queue.size();
	}
}

