package net.sodacan.core;

/**
 * <p>When an actor implements this interfact, two methods are added to the implementation. The first identifies messages that require a precheck.
 * The second method determines if the actor is (now) ready to process the message.</p>
 * <p>if this message requires an external action prior to being processed, Most commonly used 
 * to update foreign records before processing a message that requires the foreign record be up-to-date. In effect, precheck allows a request/reply
 * interaction prior to processing certain messages.</p>
 * <p>A precheck method has full access to the Actor's state and has the ability to initiate an update request. In this case, the <b>response</b> 
 * to the update request <b>should not</b> be included in precheck. The update message should never be held. But once the update(s) is (are) 
 * applied, messages awaiting precheck will now pass precheck and thus will be processed normally.</p>
 */
public interface Precheck {
	/**
	 * <p>
	 * Determine if the supplied message may need to be held.
	 * This method should not have any side-effects. It should only determine
	 *  if the message might be eligible for being held. The request and release methods
	 *  will be used by the framework to determine if and how the required state is
	 *  satisfied. Any message that returns false from this method will be
	 *  processed without any possibility of being held.
	 * </p>
	 * @param message The message being considered
	 * @return True if the message should be set aside until released.
	 */

	public boolean hold(Message message);
	/**
	 * When a messages is about to be held, the release method is called to see if the message is 
	 * clear to process or not. If not, then this method is called to send a request elsewhere to
	 * get the state of this actor into a sufficient state to be able to pass the release request.
	 * Naturally, the expected reply message, an updateMessage,  
	 * @param message
	 * @return
	 */
	public Stage request(Message message);

	/**
	 * This method is called with a message to determine if the provided message can be processed yet.
	 * @param message
	 * @return True if the message can be released for processing.
	 */
	public boolean release(Message message);
}
