package net.sodacan.core;

import java.io.Closeable;
import java.io.IOException;
import java.util.List;
import java.util.Set;

import net.sodacan.core.coordinator.HostEntry;

/**
 * <p>A Host sits at the top of the hierarchy of actorGroups and actors 
 * in a single Java application</p>.
 * <p>An instance of a Host maintains a collection of one or more actorGroups.
 * The Host handles the lifecycle of those actorGroups per instructions from the coordinator.
 * </p>
 * <p>A Host handles communication with other Hosts.
 * </p>
 * <p>An individual instance of an actor is permanently linked to a specific actorGroup
 * that contains other actors assigned to that actorGroup. An actor cannot move to another actorGroup.
 * All actorGroups in a system are known even if the number of actors can grow. While actorGroups can
 * be added, doing so is somewhat involved. actorGroup overhead is small and so it is preferred 
 * to allocate the number or actorGroups ultimately needed early in the development process.
 * </p>
 * <p>A Host does not maintain exclusive ownership of its actorGroups. In fact, it is common and
 * desirable for a actorGroup to exist on more than one Host. However, only one of the copies of a 
 * actorGroup will be considered live at any moment in time. The other copy or copies will usually 
 * be in backup (hot standby) mode. In this mode, the actorGroup is accumulating actor state changes
 * from the active actorGroup. 
 * </p>
 * <p>The default implementation of Host keeps only one copy of each actorGroup and requires no cluster coordination.
 * </p>
 * <p>The Host is not generally involved in the handling of individual messages 
 * which is done at the actorGroup level. However, the Host *is* responsible for providing a 
 * communication channel when one actorGroup needs to communicate to another actorGroup. This is needed
 * because the two actorGroups needing to communicate may be on the same Host or different Hosts. And
 * the situation can change from one moment to the next. This coordination is all the responsibility of the Host.
 * </p>
 * <p>Messages can arrive at the host from within actorGroups (from actors)
 * Or, from outside the host either from other hosts or from outside of the host
 * in the local application.
 * </p>
 * <p>In any case, it is the hosts job to act as traffic manager for messages.
 * And this process begins by determining which actorGroup the target of the message
 * belongs to. If the target actorGroup is not currently active, then it is the hosts
 * job to get the message to the host with the active target actorGroup.
 * </p>
 * <p>A small but important exception: It is common to have actorGroups active on one host send backup
 * messages to replica actorGroups on other hosts. In this case, backup messages need to
 * be routed through to actorGroups in replica mode.
 * </p>
 * <p>Interestingly, the host doesn't buffer or queue messages. The entire path from
 * the submit method to the actorGroup submission is not synchronized or locked in any way.
 * This path is careful not to update member variables, except Atomic.. fields or to take 
 * any non-thread-safe action.
 * The queuing of a message doesn't occur until it reaches the target actor. If the actorGroup containing
 * the target actor is busy, the submit will block until room is safely available for more messages.
 * </p>
 */
public interface Host extends Closeable {
	public Config getConfig();
	/**
	 * Update the host configuration by consulting with coordinator
	 */
	public void update();
	public int getHostNumber();
	public Coordinator getCoordinator();
	public void initialize();
	public Set<HostEntry> getHostEntries();
	public ActorGroup getActorGroup(HostEntry hostEntry);
	public Set<ActorGroup> getActorGroups();
	public boolean submit(Message message);
	public boolean submit(Stage messages);
	public void close() throws IOException;
}
