package net.sodacan.core;

import java.io.PrintStream;

/**
 * <p>The Sodacan configuration contains settings for specifying factories, singleton objects,
 * and static and dynamic settings.</p>
 * <p>Most configuration settings can be specified as functions. If the setting is "dynamic", 
 * the function will be called each time a value is requested. If the setting is "static",
 * the function is called once, during initialization and the result is saved and returned as
 * needed.
 * </p>
 * <p>The configuration
 * object is effectively read-only and is not thread-safe. Therefore, any of the functions
 * should be free of side-effects. In cases where side effects are required, this
 * configuration only returns a factory. In other words, the function returns a factory
 * which is used to create the target object and the factory's thread safety must be 
 * secured by the framework
 * </p>
 * <p>An exception to the thread safety rule is for global dynamic settings of which 
 * there are currently none.
 * </p>
 */
public interface Config {
	
	/**
	 * <p>Each host in the network has a unique number. This number is used to coordinate among multiple hosts in a network.
	 * The default is 1. Once assigned, a host number should be permanent. No duplicates.</p>
	 * @return An integer containing the host number
	 */
	public int getHostNumber();
	/**
	 * <p>The number of actorGroups in a network is more or less fixed. Changing this number is difficult. The number must be
	 * consistent on all hosts in the cluster. As such, this setting is maintained by the coordinator.
	 * </p>
	 * @return The number of actorGroups
	 */
	public int getActorGroups();

	public int getActorGroupReplicas();

	/**
	 * <p>The number of messages inFlight before backpressure is applied. This limit can change
	 * at any time.</p>
	 * @return
	 */
	public int getBackpressureLimit();

	public int getEviction();

	public ActorGroupAssigner getActorGroupAssigner();

	public ActorIdFactory getActorIdFactory();
	
	/**
	 * Local static
	 * Return the current number of threads 
	 * @return
	 */
	public int getActorGroupThreads();

	public Coordinator getCoordinator();

	/**
	 * <p>Return the singleton host for this application instance</p>
	 * @return
	 */
	public Host getHost();
	
	/**
	 * Create a actorGroup. 
	 * @return
	 */
	public ActorGroup createActorGroup(int actorGroupNumber);
	
//%%% ADD SENDER ACTORID For special actor.	And register sender.
//%%% listener actor (forwards directly to host without deserialization)
//%%% sender actorS
//%%% 
	/**
	 * <p>Create a new scheduler. The function is a factor for creating schedulers.
	 * </p>
	 * @return The new Scheduler subclass
	 */
	public Scheduler createScheduler();

	public int getBackpressureWaitMs();

	public int getShutdownWaitMs();

	public String getProperty(String name);
	
	/**
	 * Get the class associated with an ActorType which is either a string assigned to the actor
	 * by the ActorType annotation or the full class name.
	 * @param actorType string
	 * @return Either null or the Actor class, suitable to use to instantiate that actor.
	 */
	public Class<? extends Actor> getActorClass(String actorType);

	public Actor createActor(ActorId actorId);
	
	/**
	 * Send a list of actorType names and actor class name pairs
	 * @param out
	 */
	public void printConfig(PrintStream out);

	/**
	 * Send a list of actorType names and actor class name pairs
	 * @param out
	 */
	public void listActors(PrintStream out);
}
