package snaq.util.jclap;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * Base implementation of a command-line option.
 * 
 * @param <E> the return type of the option
 *
 * @author Giles Winstanley
 */
public abstract class Option<E>
{
  /** String denoting illegal characters for short name. */
  protected static final String ILLEGAL_SN_CHARS = "-";
  /** Limit to minimum number of occurrences required for this option. */
  protected static final int MIN_COUNT_LIMIT = 0;
  /** Limit to maximum number of occurrences required for this option (set to an arbitrary 100). */
  protected static final int MAX_COUNT_LIMIT = 100;
  /** {@code Locale} for help/error display. */
  protected static final Locale locale;
  /** Resources for internationalization. */
  protected static final ResourceBundle bundle;
  /** Short version of the option flag. */
  private String shortName;
  /** Long version of the option flag. */
  private String longName;
  /** Description of the option flag. */
  private String description;
  /** Flag to indicate whether the option requires a value to be specified. */
  private boolean requiresValue = false;
  /** Minimum number of occurrences required for this option. */
  private int minCount = MIN_COUNT_LIMIT;
  /** Maximum number of occurrences required for this option. */
  private int maxCount = MAX_COUNT_LIMIT;
  /** List of values for this option. */
  protected List<E> values = new ArrayList<>();
  /** Flag to indicate whether the option should be omitted from the usage message. */
  private boolean hideFromUsage = false;

  static
  {
    locale = CLAParser.getLocale();
    bundle = CLAParser.getResources();
  }

  /**
   * Creates a new {@code Option} instance.
   * @param shortName short name of the option (must be non-null; e.g. -?)
   * @param longName long name of the option (e.g. --help)
   * @param description helpful description of the option (printed for usage message)
   * @param requiresValue whether this option requires a subsequently-specified value
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   */
  protected Option(String shortName, String longName,
    String description, boolean requiresValue, int minCount, int maxCount)
  {
    // Perform different validation for regular Option compared to NonOptionArg.
    if (shortName == null)
      throw new IllegalArgumentException(bundle.getString("err.NullShortName"));
    if (shortName.length() != 1 || ILLEGAL_SN_CHARS.contains(shortName))
      throw new IllegalArgumentException(bundle.getString("err.BadShortName"));
    if (longName != null && longName.length() < 2)
      throw new IllegalArgumentException(bundle.getString("err.BadLongName"));
    if (minCount < MIN_COUNT_LIMIT)
      throw new IllegalArgumentException(bundle.getString("err.InvalidMinCount"));
    if (maxCount < minCount || maxCount > MAX_COUNT_LIMIT)
      throw new IllegalArgumentException(bundle.getString("err.InvalidMaxCount"));

    this.shortName = shortName;
    this.longName = longName;
    this.description = description;
    this.requiresValue = requiresValue;
    this.minCount = minCount;
    this.maxCount = maxCount;
  }

  /**
   * Creates a new {@code Option} instance.
   * @param shortName short name of the option (must be non-null; e.g. -?)
   * @param longName long name of the option (e.g. --help)
   * @param description helpful description of the option (printed for usage message)
   * @param requiresValue whether this option requires a subsequently-specified value
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   */
  protected Option(String shortName, String longName,
    String description, boolean requiresValue, boolean mandatory, boolean allowMany)
  {
    this(shortName, longName, description, requiresValue, mandatory ? 1 : 0, allowMany ? MAX_COUNT_LIMIT : 1);
  }

  @Override
  @SuppressWarnings("unchecked")
  public boolean equals(Object obj)
  {
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    final Option<E> other = (Option<E>)obj;
    if ((this.shortName == null) ? (other.shortName != null) : !this.shortName.equals(other.shortName))
      return false;
    return true;
  }

  @Override
  public int hashCode()
  {
    int hash = 3;
    hash = 59 * hash + (this.shortName != null ? this.shortName.hashCode() : 0);
    return hash;
  }

  @Override
  public String toString()
  {
    StringBuilder sb = new StringBuilder();
    sb.append("[-");
    sb.append(shortName);
    if (longName != null)
    {
      sb.append(",--");
      sb.append(longName);
    }
    sb.append(",");
    sb.append(getType().getName());
    sb.append("]");
    return sb.toString();
  }

  /**
   * Returns the class type of value this option can take.
   * @return Class instance
   */
  public abstract Class<E> getType();

  /**
   * Returns the option type, for display in the usage message.
   * This may be overridden by sub-classes to provide a more meaningful type.
   * By default this returns an appropriate type for the primitives,
   * and string type for all others.
   * @return string describing type
   */
  protected String getUsageTypeString()
  {
    String valType = getType().getName();  // e.g. "java.lang.Integer"
    valType = valType.substring(valType.lastIndexOf('.') + 1);  // e.g. "Integer"

    String usageType = null;
    try
    {
      usageType = bundle.getString("option.type." + valType);
    }
    catch (MissingResourceException mrx)
    {
      usageType = bundle.getString("option.type.unknown");
    }
    return usageType;
  }

  /**
   * Returns the mapped value of the specified argument if it exists.
   * @return list of parsed values
   */
  public List<E> getValues()
  {
    return Collections.unmodifiableList(values);
  }

  /**
   * Adds the specified value to the list of those parsed.
   * @param value the value to add to this option
   * @throws OptionException if a single-value option already has a value
   */
  protected void addValue(E value) throws OptionException
  {
    if (maxCount == 0)
      throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, this, value.toString());
    if (maxCount == 1 && !values.isEmpty())
      throw new OptionException(OptionException.Type.OPTION_HAS_VALUE, this, value.toString());
    if (values.size() >= maxCount)
      throw new OptionException(OptionException.Type.INVALID_OPTION_VALUE_COUNT, this, value.toString());
    values.add(value);
  }

  /**
   * Parses the argument string for an option value.
   * (e.g. Converts "123" to an {@code Integer} instance representing 123).
   * @param arg string argument from which a value is to be parsed
   * @throws OptionException if a problem occurs while parsing the argument string
   * @return Value of the parsed argument string
   */
  protected abstract E parseValue(String arg) throws OptionException;

  /**
   * Returns the short name of this option.
   * @return string
   */
  public String getShortName() { return shortName; }

  /**
   * Returns the long name of this option.
   * @return string
   */
  public String getLongName()
  {
    return longName;
  }

  /**
   * Returns the description text of this Option.
   * @return string
   */
  public String getDescription()
  {
    return description;
  }

  /**
   * Returns whether this option requires a value.
   * @return boolean
   */
  public boolean requiresValue()
  {
    return requiresValue;
  }

  /**
   * Returns the minimum value count for this option.
   * @return integer
   */
  public int getMinCount()
  {
    return minCount;
  }

  /**
   * Returns the maximum value count for this option.
   * @return integer
   */
  public int getMaxCount()
  {
    return maxCount;
  }

  /**
   * Returns whether this option is mandatory.
   * @return boolean
   */
  public boolean isMandatory()
  {
    return minCount > 0;
  }

  /**
   * Returns whether this option allows multiple values to be set.
   * @return boolean
   */
  public boolean isAllowMany()
  {
    return maxCount > 1;
  }

  /**
   * Sets the minimum/maximum value counts for this option.
   * @param minCount minimum count for the option
   * @param maxCount maximum count for the option
   */
  public void setMinMaxCounts(int minCount, int maxCount)
  {
    if (minCount < MIN_COUNT_LIMIT)
      throw new IllegalArgumentException(bundle.getString("err.InvalidMinCount"));
    if (maxCount < minCount || maxCount > MAX_COUNT_LIMIT)
      throw new IllegalArgumentException(bundle.getString("err.InvalidMaxCount"));
    this.minCount = minCount;
    this.maxCount = maxCount;
  }

  /**
   * Sets the flag to hide this option from the usage message.
   * @return reference to option
   */
  public Option<E> setHidden()
  {
    hideFromUsage = true;
    return this;
  }

  /**
   * Returns whether this option is hidden from the usage message.
   * @return boolean
   */
  public boolean isHidden()
  {
    return hideFromUsage;
  }
}
