package snaq.util.jclap;

/**
 * Implementation of an {@code Option} with value of type {@code Integer}.
 *
 * @author Giles Winstanley
 */
public class IntegerOption extends Option<Integer>
{
  /**
   * Creates a new {@code IntegerOption} instance.
   * @param shortName short name of the option (e.g. -?)
   * @param longName long name of the option (e.g. --help)
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   */
  public IntegerOption(String shortName, String longName, String description,
    int minCount, int maxCount)
  {
    super(shortName, longName, description, true, minCount, maxCount);
  }

  /**
   * Creates a new {@code IntegerOption} instance.
   * @param shortName short name of the option (e.g. -?)
   * @param longName long name of the option (e.g. --help)
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   */
  public IntegerOption(String shortName, String longName, String description,
    boolean mandatory, boolean allowMany)
  {
    super(shortName, longName, description, true, mandatory, allowMany);
  }

  /**
   * Parses the argument string for an option value.
   * @param arg string argument from which a value is to be parsed
   * @throws OptionException if a problem occurs while parsing the argument string
   * @return Value of the parsed argument string
   */
  @Override
  protected Integer parseValue(String arg) throws OptionException
  {
    try
    {
      return new Integer(arg);
    }
    catch (NumberFormatException e)
    {
      throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, this, arg);
    }
  }

  @Override
  public Class<Integer> getType()
  {
    return Integer.class;
  }
}
