package snaq.util.jclap;

import java.util.Collection;

/**
 * Implementation of an {@code Option} with value restricted to a
 * specified enumeration of type {@code String}.
 *
 * @author Giles Winstanley
 */
public final class EnumeratedStringOption extends EnumeratedOption<String>
{
  private boolean ignoreCase = true;

  /**
   * Creates a new {@code EnumeratedStringOption} instance.
   * @param shortName short name of the option (e.g. -t)
   * @param longName long name of the option (e.g. --type)
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @param allowedValues collection of possible values this option can take
   * @param ignoreCase whether to ignore the case in string evaluations
   */
  public EnumeratedStringOption(String shortName, String longName,
    String description, int minCount, int maxCount,
    Collection<String> allowedValues, boolean ignoreCase)
  {
    super(shortName, longName, description, minCount, maxCount, allowedValues);
  }

  /**
   * Creates a new {@code EnumeratedStringOption} instance.
   * @param shortName short name of the option (e.g. -t)
   * @param longName long name of the option (e.g. --type)
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @param allowedValues collection of possible values this option can take
   * @param ignoreCase whether to ignore the case in string evaluations
   */
  public EnumeratedStringOption(String shortName, String longName,
    String description, boolean mandatory, boolean allowMany,
    Collection<String> allowedValues, boolean ignoreCase)
  {
    super(shortName, longName, description, mandatory, allowMany, allowedValues);
  }

  /**
   * Parses the argument string for an option value.
   * @param arg string argument from which a value is to be parsed
   * @throws OptionException if a problem occurs while parsing the argument string
   * @return Value of the parsed argument string
   */
  @Override
  protected String parseValue(String arg) throws OptionException
  {
    for (String s : allowedValues)
    {
      if ( (ignoreCase && s.equalsIgnoreCase(arg)) || (!ignoreCase && s.equals(arg)) )
        return arg;
    }
    throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, this, arg);
  }

  @Override
  public Class<String> getType()
  {
    return String.class;
  }

  /**
   * Returns the default version of the string denoting the values that
   * can be assigned to this option ()
   */
  @Override
  public String getAllowedValuesString()
  {
    return getAllowedValuesString("\"", ", ");
	}
}
