package snaq.util.jclap;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.text.MessageFormat;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Utility class to provide support for command-line options parsing.
 * Has support for POSIX-style short-form (e.g.&nbsp;{@code -v}) and GNU-style
 * long-form (e.g.&nbsp;{@code --verbose}) options (short-form is mandatory,
 * long-form optional).
 * Option processing can be explicitly terminated by the argument
 * &quot;{@code --}&quot;, to allow (non-option) arguments to be specified.
 * <p>
 * All options must have at least a short name, and may also have a long name
 * (recommended).
 * An option that requires no value is considered a boolean/flag.
 * Options that require values have the value specified immediately after the
 * option name on the command-line.
 * For example, an option to display a help message might be specified
 * with the short name {@code ?} and the long name {@code help}
 * (abbreviated to {@code [-?,--help]} in the usage message).
 * An option for a size value might be specified as {@code [-s,--size]}
 * and require a value parameter, which may be specified in any of these forms:
 * <ul>
 * <li>{@code -s 123}</li>
 * <li>{@code -s=123}</li>
 * <li>{@code -s:123}</li>
 * <li>{@code /s 123}</li>
 * <li>{@code /s=123}</li>
 * <li>{@code /s:123}</li>
 * <li>{@code --size 123}</li>
 * <li>{@code --size=123}</li>
 * <li>{@code --size:123}</li>
 * </ul>
 * <p>
 * Combining these the command-line may look like these examples, depending
 * how the application has been configured:
 * <pre>
 *     java AppName [-?,--help] [--size &lt;integer&gt;]
 *     java AppName -w,--width &lt;integer&gt; -h,--height &lt;integer&gt; [-?]
 * </pre>
 * where options in [brackets] are optional, and others are mandatory.
 * <p>
 * Here are some of the features of JCLAP:
 * <ul>
 * <li>Option short names are mandatory, and long names optional
 * (as per POSIX standard, but retaining GNU enhancements).</li>
 * <li>Uses a single internally-typed {@code OptionException} type for signalling parsing/retrieval problems.</li>
 * <li>Supports enumerated option types (String, Integer).</li>
 * <li>Supports filtered string option types.</li>
 * <li>Supports file/folder option types (using filtered-string feature).</li>
 * <li>Supports single-hyphen specification for reading from STDIN.</li>
 * <li>Support for automatic usage printing (both long &amp; short styles).</li>
 * <li>Basic internationalization support.</li>
 * </ul>
 * <p>
 * Automatic usage printing can be accessed via the {@code printUsage(...)}
 * method, allowing printing to a specified {@code PrintStream} in either long
 * or short versions.
 * The options are printed in the same order in which they were added to the
 * {@code CLAParser} instance. The command used to launch the application
 * can either be user-specified, or can be deduced automatically if required.
 * An additional usage suffix string can also be specified if required,
 * allowing use of non-option arguments.
 * <p>
 * Internationalization support is included for a few locales
 * (en/fr/de/es/pt), and may be extended in future. If required users may
 * author their own locale property files to reference.
 * <p>
 * For an example of how to use the CLAParser class, visit the website.
 *
 * @see <a href="http://www.snaq.net/" target="_blank">JCLAP website</a>
 * @author Giles Winstanley
 */
public final class CLAParser
{
  private static final boolean DEBUG = false;
  private static final int INIT_MAP_SIZE = 8;
  /** {@code Locale} for help/error display. */
  private static final Locale locale;
  /** Resources for internationalisation. */
  private static final ResourceBundle bundle;
  /** Resources for internationalisation (for base locale). */
  private static final ResourceBundle baseBundle;
  /**
   * List of options. While a map would seem to make more sense, using a list
   * adds insignificant overhead for the few elements used in this scenario,
   * and adds the benefit of easily maintaining the order in which the options
   * were specified.
   */
  private final List<Option> options = new ArrayList<>(INIT_MAP_SIZE);
  /** List of non-option arguments from the command-line. */
  private List<String> nonOptionArgs = null;

  static
  {
    locale = Locale.getDefault();
    bundle = ResourceBundle.getBundle(CLAParser.class.getName(), locale);
    baseBundle = ResourceBundle.getBundle(CLAParser.class.getName(), Locale.ENGLISH);
  }

  /**
   * Creates a new {@code CLAParser} instance, using the default locale.
   */
  public CLAParser()
  {
  }

  /** Returns the locale. */
  static Locale getLocale()
  {
    return locale;
  }

  /** Returns the ResourceBundle for localization. */
  static ResourceBundle getResources()
  {
    return bundle;
  }

  /** Returns the base ResourceBundle for default localization. */
  static ResourceBundle getBaseResources()
  {
    return baseBundle;
  }

  /**
   * Returns whether a single {@code -} argument was found.
   * This is typically used to specify that an application should read from
   * STDIN or write to STDOUT. The argument is written to the non-option
   * arguments, so this can be tested independently, but this is just a
   * convenience method to do the same.
   * @return true if solitary hyphen was parsed, false otherwise
   */
  public boolean parsedSolitaryHyphen()
  {
    for (String s : nonOptionArgs)
    {
      if (s.equals("-"))
        return true;
    }
    return false;
  }

  /**
   * Adds the specified {@code Option} to the list of those to be parsed
   * from the command-line arguments.
   * @param <T> the return type of the option
   * @param opt {@code Option} instance to add
   * @throws IllegalArgumentException if either short/long name of the specified
   *   {@code Option} is already in use
   * @return the {@code Option} instance that was added
   */
  public <T> Option<T> addOption(Option<T> opt)
  {
    // Check whether option's short/long name are already in use.
    for (Option o : options)
    {
      String optSN = opt.getShortName();
      String optLN = opt.getLongName();
      String oLN = o.getLongName();
      if (o.getShortName().equals(optSN))
        throw new IllegalArgumentException(MessageFormat.format(bundle.getString("err.OptionNameInUse"), optSN));
      if (oLN != null && optLN != null && oLN.equals(optLN))
        throw new IllegalArgumentException(MessageFormat.format(bundle.getString("err.OptionNameInUse"), optLN));
    }
    // Add to collection and return.
    options.add(opt);
    return opt;
  }

  /**
   * Hides the specified {@code Option} from being printed in the usage message.
   * @param optionName short/long name of option to &quot;hide&quot;
   * @throws OptionException if the specified option can't be found
   */
  public void setHidden(String optionName) throws OptionException
  {
    Option opt = getOptionByShortName(optionName);
    if (opt == null)
      opt = getOptionByLongName(optionName);
    if (opt == null)
      throw new OptionException(OptionException.Type.UNKNOWN_OPTION, optionName);
    opt.setHidden();
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Boolean}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Boolean> addBooleanOption(String shortName, String longName, String description, int minCount, int maxCount)
  {
    return addOption(new BooleanOption(shortName, longName, description, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Boolean}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Boolean> addBooleanOption(String shortName, String longName, String description, boolean allowMany)
  {
    return addOption(new BooleanOption(shortName, longName, description, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Boolean}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Boolean> addBooleanOption(String shortName, String longName, String description, int maxCount)
  {
    return addOption(new BooleanOption(shortName, longName, description, 0, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Boolean}
   * (single value allowed).
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @return the new {@code Option} instance
   */
  public Option<Boolean> addBooleanOption(String shortName, String longName, String description)
  {
    return addBooleanOption(shortName, longName, description, false);
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Boolean}.
   * @param shortName short name of the option
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Boolean> addBooleanOption(String shortName, boolean allowMany)
  {
    return addOption(new BooleanOption(shortName, null, null, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Boolean}.
   * @param shortName short name of the option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Boolean> addBooleanOption(String shortName, int maxCount)
  {
    return addOption(new BooleanOption(shortName, null, null, 0, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Boolean}
   * (single value allowed).
   * @param shortName short name of the option
   * @return the new {@code Option} instance
   */
  public Option<Boolean> addBooleanOption(String shortName)
  {
    return addBooleanOption(shortName, false);
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Integer}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Integer> addIntegerOption(String shortName, String longName, String description, boolean mandatory, boolean allowMany)
  {
    return addOption(new IntegerOption(shortName, longName, description, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Integer}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Integer> addIntegerOption(String shortName, String longName, String description, int minCount, int maxCount)
  {
    return addOption(new IntegerOption(shortName, longName, description, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Integer}.
   * @param shortName short name of the option
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Integer> addIntegerOption(String shortName, boolean mandatory, boolean allowMany)
  {
    return addOption(new IntegerOption(shortName, null, null, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Integer}.
   * @param shortName short name of the option
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Integer> addIntegerOption(String shortName, int minCount, int maxCount)
  {
    return addOption(new IntegerOption(shortName, null, null, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Long}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Long> addLongOption(String shortName, String longName, String description, boolean mandatory, boolean allowMany)
  {
    return addOption(new LongOption(shortName, longName, description, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Long}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Long> addLongOption(String shortName, String longName, String description, int minCount, int maxCount)
  {
    return addOption(new LongOption(shortName, longName, description, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Long}.
   * @param shortName short name of the option
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Long> addLongOption(String shortName, boolean mandatory, boolean allowMany)
  {
    return addOption(new LongOption(shortName, null, null, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Long}.
   * @param shortName short name of the option
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Long> addLongOption(String shortName, int minCount, int maxCount)
  {
    return addOption(new LongOption(shortName, null, null, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Double}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Double> addDoubleOption(String shortName, String longName, String description, boolean mandatory, boolean allowMany)
  {
    return addOption(new DoubleOption(shortName, longName, description, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Double}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Double> addDoubleOption(String shortName, String longName, String description, int minCount, int maxCount)
  {
    return addOption(new DoubleOption(shortName, longName, description, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Double}.
   * @param shortName short name of the option
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Double> addDoubleOption(String shortName, boolean mandatory, boolean allowMany)
  {
    return addOption(new DoubleOption(shortName, null, null, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Double}.
   * @param shortName short name of the option
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Double> addDoubleOption(String shortName, int minCount, int maxCount)
  {
    return addOption(new DoubleOption(shortName, null, null, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Float}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Float> addFloatOption(String shortName, String longName, String description, boolean mandatory, boolean allowMany)
  {
    return addOption(new FloatOption(shortName, longName, description, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Float}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Float> addFloatOption(String shortName, String longName, String description, int minCount, int maxCount)
  {
    return addOption(new FloatOption(shortName, longName, description, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Float}.
   * @param shortName short name of the option
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<Float> addFloatOption(String shortName, boolean mandatory, boolean allowMany)
  {
    return addOption(new FloatOption(shortName, null, null, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code Float}.
   * @param shortName short name of the option
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<Float> addFloatOption(String shortName, int minCount, int maxCount)
  {
    return addOption(new FloatOption(shortName, null, null, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<String> addStringOption(String shortName, String longName, String description, boolean mandatory, boolean allowMany)
  {
    return addOption(new StringOption(shortName, longName, description, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<String> addStringOption(String shortName, String longName, String description, int minCount, int maxCount)
  {
    return addOption(new StringOption(shortName, longName, description, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String}
   * (single value allowed, non-mandatory).
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @return the new {@code Option} instance
   */
  public Option<String> addStringOption(String shortName, String longName, String description)
  {
    return addOption(new StringOption(shortName, longName, description, false, false));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String}
   * (no long name, no description).
   * @param shortName short name of the option
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<String> addStringOption(String shortName, boolean mandatory, boolean allowMany)
  {
    return addOption(new StringOption(shortName, null, null, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String}
   * (no long name, no description).
   * @param shortName short name of the option
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<String> addStringOption(String shortName, int minCount, int maxCount)
  {
    return addOption(new StringOption(shortName, null, null, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String} which
   * refers to a file/folder.
   * For example, use this method to:
   * <ul>
   * <li>Allow specification of a existing or new file/folder.</li>
   * </ul>
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<String> addFileOption(String shortName, String longName,
    String description, boolean mandatory, boolean allowMany)
  {
    FileOption opt = new FileOption(shortName, longName, description, mandatory, allowMany);
    FileOption.FileFilter.AcceptExistance eType = FileOption.FileFilter.AcceptExistance.ACCEPT_ALL;
    FileOption.FileFilter.AcceptFileType fType = FileOption.FileFilter.AcceptFileType.ACCEPT_ALL;
    FileOption.Filter filter = new FileOption.FileFilter(eType, fType);
    opt.setFilter(filter);
    return addOption(opt);
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String} which
   * refers to a new or existing file/folder.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<String> addFileNewOption(String shortName, String longName,
    String description, boolean mandatory, boolean allowMany)
  {
    FileOption opt = new FileOption(shortName, longName, description, mandatory, allowMany);
    FileOption.FileFilter.AcceptExistance eType = FileOption.FileFilter.AcceptExistance.ACCEPT_NON_EXISTING;
    FileOption.FileFilter.AcceptFileType fType = FileOption.FileFilter.AcceptFileType.ACCEPT_ALL;
    FileOption.Filter filter = new FileOption.FileFilter(eType, fType);
    opt.setFilter(filter);
    return addOption(opt);
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String} which
   * refers to an existing file (which is not a folder).
   * For example, use this method to:
   * <ul>
   * <li>Enforce specification of an existing file (which is not a folder)</li>
   * </ul>
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<String> addFileExistingOption(String shortName, String longName,
    String description, int minCount, int maxCount)
  {
    FileOption opt = new FileOption(shortName, longName, description, minCount, maxCount);
    FileOption.FileFilter.AcceptExistance eType = FileOption.FileFilter.AcceptExistance.ACCEPT_EXISTING;
    FileOption.FileFilter.AcceptFileType fType = FileOption.FileFilter.AcceptFileType.ACCEPT_FILE;
    FileOption.Filter filter = new FileOption.FileFilter(eType, fType);
    opt.setFilter(filter);
    return addOption(opt);
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String} which
   * refers to an existing file (which is not a folder).
   * For example, use this method to:
   * <ul>
   * <li>Enforce specification of an existing file (which is not a folder)</li>
   * </ul>
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<String> addFileExistingOption(String shortName, String longName,
    String description, boolean mandatory, boolean allowMany)
  {
    return addFileExistingOption(shortName, longName, description, mandatory ? 1 : 0, allowMany ? Option.MAX_COUNT_LIMIT : 1);
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String} which
   * refers to an existing folder.
   * For example, use this method to enforce specification of an existing folder.
   * To allow specification of a new folder (one that does not yet exist),
   * use the {@code addFileOption()} method instead.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<String> addFolderExistingOption(String shortName, String longName,
    String description, int minCount, int maxCount)
  {
    FileOption opt = new FileOption(shortName, longName, description, minCount, maxCount);
    FileOption.FileFilter.AcceptExistance eType = FileOption.FileFilter.AcceptExistance.ACCEPT_EXISTING;
    FileOption.FileFilter.AcceptFileType fType = FileOption.FileFilter.AcceptFileType.ACCEPT_DIR;
    FileOption.Filter filter = new FileOption.FileFilter(eType, fType);
    opt.setFilter(filter);
    return addOption(opt);
  }

  /**
   * Convenience method to add an {@code Option} of type {@code String} which
   * refers to an existing folder.
   * For example, use this method to enforce specification of an existing folder.
   * To allow specification of a new folder (one that does not yet exist),
   * use the {@code addFileOption()} method instead.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<String> addFolderExistingOption(String shortName, String longName,
    String description, boolean mandatory, boolean allowMany)
  {
    return addFolderExistingOption(shortName, longName, description, mandatory ? 1 : 0, allowMany ? Option.MAX_COUNT_LIMIT : 1);
  }

  /**
   * Convenience method to add an {@code Option} of enumerated {@code String} type.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @param allowedValues array of string values allowed by this enumerated option
   * @param ignoreCase whether to ignore the case of string values
   * @return the new {@code Option} instance
   */
  public Option<String> addEnumStringOption(String shortName, String longName, String description, boolean mandatory, boolean allowMany, String[] allowedValues, boolean ignoreCase)
  {
    Collection<String> x = new ArrayList<>();
    x.addAll(Arrays.asList(allowedValues));
    return addOption(new EnumeratedStringOption(shortName, longName, description, mandatory, allowMany, x, ignoreCase));
  }

  /**
   * Convenience method to add an {@code Option} of enumerated {@code String} type.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @param allowedValues array of string values allowed by this enumerated option
   * @param ignoreCase whether to ignore the case of string values
   * @return the new {@code Option} instance
   */
  public Option<String> addEnumStringOption(String shortName, String longName, String description, int minCount, int maxCount, String[] allowedValues, boolean ignoreCase)
  {
    Collection<String> x = new ArrayList<>();
    x.addAll(Arrays.asList(allowedValues));
    return addOption(new EnumeratedStringOption(shortName, longName, description, minCount, maxCount, x, ignoreCase));
  }

  /**
   * Convenience method to add an {@code Option} of enumerated {@code String} type
   * (case-insensitive string matching).
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @param allowedValues array of string values allowed by this enumerated option
   * @return the new {@code Option} instance
   */
  public Option<String> addEnumStringOption(String shortName, String longName, String description, boolean mandatory, boolean allowMany, String[] allowedValues)
  {
    return addEnumStringOption(shortName, longName, description, mandatory, allowMany, allowedValues, true);
  }

  /**
   * Convenience method to add an {@code Option} of enumerated {@code String} type
   * (single value allowed, non-mandatory).
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param allowedValues array of string values allowed by this enumerated option
   * @return the new {@code Option} instance
   */
  public Option<String> addEnumStringOption(String shortName, String longName, String description, String[] allowedValues)
  {
    return addEnumStringOption(shortName, longName, description, false, false, allowedValues);
  }

  /**
   * Convenience method to add an {@code Option} of enumerated {@code Integer} type.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @param allowedValues array of string values allowed by this enumerated option
   * @return the new {@code Option} instance
   */
  public Option<Integer> addEnumIntegerOption(String shortName, String longName, String description, boolean mandatory, boolean allowMany, int[] allowedValues)
  {
    Collection<Integer> x = new ArrayList<>();
    for (Integer s : allowedValues)
      x.add(s);
    return addOption(new EnumeratedIntegerOption(shortName, longName, description, mandatory, allowMany, x));
  }

  /**
   * Convenience method to add an {@code Option} of enumerated {@code Integer} type
   * (single value allowed, non-mandatory).
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param allowedValues array of string values allowed by this enumerated option
   * @return the new {@code Option} instance
   */
  public Option<Integer> addEnumIntegerOption(String shortName, String longName, String description, int[] allowedValues)
  {
    return addEnumIntegerOption(shortName, longName, description, false, false, allowedValues);
  }

  /**
   * Convenience method to add an {@code Option} of type {@code LocalDate}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<LocalDate> addLocalDateOption(String shortName, String longName, String description, boolean mandatory, boolean allowMany)
  {
    return addOption(new LocalDateOption(shortName, longName, description, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code LocalDate}.
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<LocalDate> addLocalDateOption(String shortName, String longName, String description, int minCount, int maxCount)
  {
    return addOption(new LocalDateOption(shortName, longName, description, minCount, maxCount));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code LocalDate}
   * (single value allowed, non-mandatory).
   * @param shortName short name of the option
   * @param longName long name of the option
   * @param description helpful description of the option (printed for usage message)
   * @return the new {@code Option} instance
   */
  public Option<LocalDate> addLocalDateOption(String shortName, String longName, String description)
  {
    return addOption(new LocalDateOption(shortName, longName, description, false, false));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code LocalDate}
   * (no long name, no description).
   * @param shortName short name of the option
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   * @return the new {@code Option} instance
   */
  public Option<LocalDate> addLocalDateOption(String shortName, boolean mandatory, boolean allowMany)
  {
    return addOption(new LocalDateOption(shortName, null, null, mandatory, allowMany));
  }

  /**
   * Convenience method to add an {@code Option} of type {@code LocalDate}
   * (no long name, no description).
   * @param shortName short name of the option
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   * @return the new {@code Option} instance
   */
  public Option<LocalDate> addLocalDateOption(String shortName, int minCount, int maxCount)
  {
    return addOption(new LocalDateOption(shortName, null, null, minCount, maxCount));
  }

  /**
   * Returns the {@code Option} with the specified short name, or null if not found.
   * @param shortName short name of the option
   */
  private Option getOptionByShortName(String shortName)
  {
    if (shortName == null || shortName.equals(""))
      throw new IllegalArgumentException();
    for (Option o : options)
    {
      if (shortName.equals(o.getShortName()))
        return o;
    }
    return null;
  }

  /**
   * Returns the {@code Option} with the specified short name, or null if not found.
   * @param shortName short name of the option
   */
  private Option getOptionByShortName(char shortName)
  {
    return getOptionByShortName(new String(new char[]{shortName}));
  }

  /**
   * Returns the {@code Option} with the specified short name, or null if not found.
   * @param longName long name of the option
   */
  private Option getOptionByLongName(String longName)
  {
    if (longName == null || longName.equals(""))
      throw new IllegalArgumentException();
    for (Option o : options)
    {
      if (longName.equals(o.getLongName()))
        return o;
    }
    return null;
  }

  /**
   * Returns the {@code Option} with the specified name (either short or long).
   * @param <T> the return type of the option
   * @param optionName option name for which to get value (either short/long)
   * @param type class type for option value compatability check
   * @return Option instance
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  @SuppressWarnings("unchecked")
  public <T> Option<T> getOption(String optionName, Class<T> type) throws OptionException
  {
    if (optionName == null)
      throw new OptionException(OptionException.Type.UNKNOWN_OPTION, optionName);
    Option opt = null;
    if (optionName.length() == 1)
      opt = getOptionByShortName(optionName);
    if (opt == null)
      opt = getOptionByLongName(optionName);
    if (opt == null)
      throw new OptionException(OptionException.Type.UNKNOWN_OPTION, optionName);
    if (!type.equals(opt.getType()))
      throw new OptionException(OptionException.Type.INVALID_RETRIEVAL_TYPE, optionName);
    return opt;
  }

  /**
   * Returns all the {@code Option} instances registered with the parser.
   * @return list of options
   */
  public final List<Option> getOptions()
  {
    return Collections.unmodifiableList(options);
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * @param <T> the return type of the option
   * @param opt option for which to get values
   * @throws OptionException if {@code opt} is null, cannot be found, or is of the wrong type
   * @return A list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   */
  public <T> List<T> getOptionValues(Option<T> opt) throws OptionException
  {
    if (opt == null)
      throw new OptionException(OptionException.Type.UNKNOWN_OPTION, opt);
    return Collections.unmodifiableList(opt.getValues());
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * This method performs type-checking of the {@code Option}.
   * @param <T> the return type of the option
   * @param optionName option name for which to get value (either short/long)
   * @param type class type for option value compatability check
   * @return list of option values
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public <T> List<T> getOptionValues(String optionName, Class<T> type) throws OptionException
  {
    return getOptionValues(getOption(optionName, type));
  }

  /**
   * Convenience method to return the parsed value of the specified option,
   * or null if the option was not set.
   * This method is only usable with options that cannot take multiple values.
   * @param <T> the return type of the option
   * @param opt option for which to get value
   * @return option value
   * @throws OptionException if {@code opt} is null, cannot be found, or is of the wrong type
   */
  public <T> T getOptionValue(Option<T> opt) throws OptionException
  {
    if (opt == null)
      throw new OptionException(OptionException.Type.UNKNOWN_OPTION, opt);
    if (opt.isAllowMany())
      throw new OptionException(OptionException.Type.INVALID_RETRIEVAL_TYPE, opt);
    List<T> vals = getOptionValues(opt);
    return vals.isEmpty() ? null : vals.get(0);
  }

  /**
   * Convenience method to return the parsed value of the specified option,
   * or the specified default value if the option was not set.
   * This method is only usable with options that cannot take multiple values.
   * @param <T> the return type of the option
   * @param opt option for which to get value
   * @param def default value to use if the option was not set
   * @return option value
   * @throws OptionException if {@code opt} is null, cannot be found, or is of the wrong type
   */
  @SuppressWarnings("unchecked")
  public <T> T getOptionValue(Option<T> opt, T def) throws OptionException
  {
    if (opt == null)
      throw new OptionException(OptionException.Type.UNKNOWN_OPTION, opt);
    if (opt.isAllowMany())
      throw new OptionException(OptionException.Type.INVALID_RETRIEVAL_TYPE, opt);

    // If EnumeratedOption, check that default value is valid.
    // This is checked regardless of parsed values to help avoid problems
    // at development time.
    if (opt instanceof EnumeratedOption)
    {
      EnumeratedOption eo = (EnumeratedOption)opt;
      if (!eo.isValueValid(def))
        throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt, def.toString());
    }

    List<T> vals = getOptionValues(opt);
    return vals.isEmpty() ? def : vals.get(0);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or the specified default value if the option was not set.
   * This method should only be used with typed options that cannot take
   * multiple values.
   * @param <T> the return type of the option
   * @param optionName name of the option
   * @param type class type for option value compatability check
   * @param def default value to use if the option was not set
   * @return option value, or default value if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public <T> T getOptionValue(String optionName, Class<T> type, T def) throws OptionException
  {
    return getOptionValue(getOption(optionName, type), def);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or null if the option was not set,
   * equivalent to {@code getOptionValue(optionName, type, null)}.
   * This method should only be used with typed options that cannot take
   * multiple values.
   * @param <T> the return type of the option
   * @param optionName name of the option
   * @param type class type for option value compatability check
   * @return option value
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public <T> T getOptionValue(String optionName, Class<T> type) throws OptionException
  {
    return getOptionValue(optionName, type, null);
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * @param optionName name of the option
   * @return list of option values
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public List<Boolean> getBooleanOptionValues(String optionName) throws OptionException
  {
    return getOptionValues(optionName, Boolean.class);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or the specified default value if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @param def default value to use if the option was not set
   * @return option value, or default value if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Boolean getBooleanOptionValue(String optionName, Boolean def) throws OptionException
  {
    return getOptionValue(optionName, Boolean.class, def);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or {@code false} if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * NOTE: this method operates slightly differently compared to the other
   * {@code getXOptionValue(String)} methods, in that when the option is not
   * defined, it returned {@code Boolean.FALSE} instead of {@code null}.
   * @param optionName name of the option
   * @return option value, or null if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Boolean getBooleanOptionValue(String optionName) throws OptionException
  {
    return getOptionValue(optionName, Boolean.class, Boolean.FALSE);
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * @param optionName name of the option
   * @return list of option values
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public List<Integer> getIntegerOptionValues(String optionName) throws OptionException
  {
    return getOptionValues(optionName, Integer.class);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or the specified default value if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @param def default value to use if the option was not set
   * @return option value, or default value if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Integer getIntegerOptionValue(String optionName, Integer def) throws OptionException
  {
    return getOptionValue(optionName, Integer.class, def);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or null if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @return option value, or null if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Integer getIntegerOptionValue(String optionName) throws OptionException
  {
    return getOptionValue(optionName, Integer.class);
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * @param optionName name of the option
   * @return list of option values
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public List<Long> getLongOptionValues(String optionName) throws OptionException
  {
    return getOptionValues(optionName, Long.class);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or the specified default value if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @param def default value to use if the option was not set
   * @return option value, or default value if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Long getLongOptionValue(String optionName, Long def) throws OptionException
  {
    return getOptionValue(optionName, Long.class, def);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or null if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @return option value, or null if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Long getLongOptionValue(String optionName) throws OptionException
  {
    return getOptionValue(optionName, Long.class);
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * @param optionName name of the option
   * @return list of option values
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public List<Double> getDoubleOptionValues(String optionName) throws OptionException
  {
    return getOptionValues(optionName, Double.class);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or the specified default value if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @param def default value to use if the option was not set
   * @return option value, or default value if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Double getDoubleOptionValue(String optionName, Double def) throws OptionException
  {
    return getOptionValue(optionName, Double.class, def);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or null if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @return option value, or null if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Double getDoubleOptionValue(String optionName) throws OptionException
  {
    return getOptionValue(optionName, Double.class);
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * @param optionName name of the option
   * @return list of option values
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public List<Float> getFloatOptionValues(String optionName) throws OptionException
  {
    return getOptionValues(optionName, Float.class);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or the specified default value if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @param def default value to use if the option was not set
   * @return option value, or default value if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Float getFloatOptionValue(String optionName, Float def) throws OptionException
  {
    return getOptionValue(optionName, Float.class, def);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or null if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @return option value, or null if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public Float getFloatOptionValue(String optionName) throws OptionException
  {
    return getOptionValue(optionName, Float.class);
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * @param optionName name of the option
   * @return list of option values
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public List<String> getStringOptionValues(String optionName) throws OptionException
  {
    return getOptionValues(optionName, String.class);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or the specified default value if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @param def default value to use if the option was not set
   * @return option value, or default value if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public String getStringOptionValue(String optionName, String def) throws OptionException
  {
    return getOptionValue(optionName, String.class, def);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or null if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @return option value, or null if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public String getStringOptionValue(String optionName) throws OptionException
  {
    return getOptionValue(optionName, String.class);
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * @param optionName name of the option
   * @return list of option values
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public List<LocalDate> getLocalDateOptionValues(String optionName) throws OptionException
  {
    return getOptionValues(optionName, LocalDate.class);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or the specified default value if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @param def default value to use if the option was not set
   * @return option value, or default value if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public LocalDate getLocalDateOptionValue(String optionName, LocalDate def) throws OptionException
  {
    return getOptionValue(optionName, LocalDate.class, def);
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or null if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @return option value, or null if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public LocalDate getLocalDateOptionValue(String optionName) throws OptionException
  {
    return getOptionValue(optionName, LocalDate.class);
  }

  /**
   * Returns a list of all the parsed values for the specified option,
   * or an empty list if the option was not set.
   * @param optionName name of the option
   * @return list of option values
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public List<File> getFileOptionValues(String optionName) throws OptionException
  {
    List<String> ov = getOptionValues(optionName, String.class);
    List<File> list = new ArrayList<>();
    try
    {
      for (String s : ov)
        list.add(new File(s).getCanonicalFile());
    }
    catch (IOException ex)
    {
      Option<String> opt = getOption(optionName, String.class);
      throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt);
    }
    return list;
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or the specified default value if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @param def default value to use if the option was not set
   * @return option value, or default value if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public File getFileOptionValue(String optionName, File def) throws OptionException
  {
    String ov = getOptionValue(optionName, String.class);
    try
    {
      return (ov == null) ? def : new File(ov).getCanonicalFile();
    }
    catch (IOException ex)
    {
      Option<String> opt = getOption(optionName, String.class);
      throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt, ov);
    }
  }

  /**
   * Convenience method to return the parsed value of the named option,
   * or null if the option was not set. This method
   * should only be used with options that cannot take multiple values.
   * @param optionName name of the option
   * @return option value, or null if option is unset
   * @throws OptionException if {@code optionName} is null, cannot be found, or is of the wrong type
   */
  public File getFileOptionValue(String optionName) throws OptionException
  {
    String ov = getOptionValue(optionName, String.class);
    try
    {
      return (ov == null) ? null : new File(ov).getCanonicalFile();
    }
    catch (IOException ex)
    {
      Option<String> opt = getOption(optionName, String.class);
      throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt, ov);
    }
  }

  /**
   * Returns a list of the non-option command-line arguments.
   * Note that this list includes all arguments that are not related to options,
   * and as a result may include &quot;orphan&quot; arguments (in between
   * correctly specified options) in addition to the expected arguments that
   * are specified after all the options.
   * <p>For example:
   * <pre>
   *     java DemoApp -i 10 -f 1.2 foo -s bar fu bar
   * </pre>
   * <p>for an application with value-taking options {@code i}/{@code f}/{@code s},
   * would return [{@code foo},{@code fu},{@code bar}] from this method.
   * @return A list of the command-line string arguments that were not parsed as options.
   */
  public List<String> getNonOptionArguments()
  {
    return Collections.unmodifiableList(nonOptionArgs);
  }

  /**
   * Extract the option-mapped and unmapped arguments from the given array of
   * command-line arguments, using the specified locale.
   * @param args command-line arguments (as passed into {@code main} method)
   * @throws OptionException if an problem is encountered parsing the specified arguments
   */
  @SuppressWarnings("unchecked")
  public void parseOrig(String[] args) throws OptionException
  {
    // Create list to hold non-option arguments found.
    List<String> spareArgs = new ArrayList<>();
    // Create list of arguments for easier processing.
    List<String> list = new ArrayList<>(args.length);
    list.addAll(Arrays.asList(args));

    // Declare flag to signal end of option processing.
    boolean eoo = false;
    // Loop over arguments to process them.
    // A ListIterator is used to provide easy access to next argument.
    for (ListIterator<String> iter = list.listIterator(); iter.hasNext();)
    {
      // Get next argument to process.
      String arg = iter.next();

      // Check for end of options.
      if (eoo)
      {
        spareArgs.add(arg);
        continue;
      }

      // Find position of '=' in argument (for possible value assignment).
      int posEqual = arg.indexOf("=", 1);

      // Process options from argument.
      if (arg.startsWith("-"))  // Option specified.
      {
        if (arg.startsWith("--"))  // Long name specified.
        {
          if (arg.equals("--"))  // End-of-options specified.
          {
            eoo = true;
            continue;  // Continue with next argument.
          }
          // Process as long name option (with possible value specified).
          String ln = (posEqual > 2) ? arg.substring(2, posEqual) : arg.substring(2);
          String val = (posEqual > 2) ? arg.substring(posEqual + 1, arg.length()) : null;
          Option opt = getOptionByLongName(ln);
          if (opt == null)
            throw new OptionException(OptionException.Type.UNKNOWN_OPTION, ln);
          // Assign value to option if required.
          if (opt.requiresValue())
          {
            // If value is currently null, then read value from next argument.
            if (val == null)
            {
              if (!iter.hasNext())
                throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt, "null");
              val = iter.next();
            }
            opt.addValue(opt.parseValue(val));
          }
          else  // otherwise if no value required it must be a boolean.
          {
            opt.addValue(Boolean.TRUE);
          }
        }
        else  // Starts with "-"; short name specified.
        {
          String optNames = (posEqual > 1) ? arg.substring(1, posEqual) : arg.substring(1);
          if (optNames.length() > 1)  // Found >1 short name.
          {
            // Loop over short names checking for concatenated flags.
            for (int i = 0; i < optNames.length(); i++)
            {
              char snc = optNames.charAt(i);
              Option opt = getOptionByShortName(snc);
              if (opt == null)  // Unknown flag option
                throw new OptionException(OptionException.Type.UNKNOWN_FLAG, optNames, snc);
              if (opt.requiresValue())  // Value-requiring option amongst flags
                throw new OptionException(OptionException.Type.NOT_FLAG, optNames, snc);
              if (DEBUG) System.err.printf("Setting flag option -%s\n", opt.getShortName());
              opt.addValue(Boolean.TRUE);
            }
          }
          else  // Found 1 short name.
          {
            String sn = arg.substring(1, 2);
            String val = (posEqual > 1) ? arg.substring(posEqual + 1, arg.length()) : null;
            Option opt = getOptionByShortName(sn);
            if (opt == null)
                throw new OptionException(OptionException.Type.UNKNOWN_FLAG, sn);
            if (opt.requiresValue())
            {
              // If value is currently null, then read value from next argument.
              if (val == null)
              {
                if (!iter.hasNext())
                  throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt, null);
                val = iter.next();
              }
              if (DEBUG) System.err.printf("Setting option -%s to value %s\n", opt.getShortName(), val);
              opt.addValue(opt.parseValue(val));
            }
            else  // otherwise if no value required it must be a boolean.
            {
              if (DEBUG) System.err.printf("Setting flag option -%s\n", opt.getShortName());
              opt.addValue(Boolean.TRUE);
            }

          }
        }
      }
      else  // Non-option argument found.
      {
        spareArgs.add(arg);
      }
    }

    // Save non-option arguments.
    nonOptionArgs = spareArgs;

    // Finally, check for value-requiring options that don't conform.
    for (Option opt : options)
    {
      int valCount = opt.getValues().size();
      if (opt.isMandatory() && opt.getValues().isEmpty())
        throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt, null);
      if (valCount < opt.getMinCount() || valCount > opt.getMaxCount())
        throw new OptionException(OptionException.Type.INVALID_OPTION_COUNT, opt, null);
    }
  }

  /**
   * Extract the option-mapped and unmapped arguments from the given array of
   * command-line arguments, using the specified locale.
   * @param args command-line arguments (as passed into {@code main} method)
   * @throws OptionException if an problem is encountered parsing the specified arguments
   */
  @SuppressWarnings("unchecked")
  public void parse(String[] args) throws OptionException
  {
    // Create list to hold non-option arguments found.
    List<String> spareArgs = new ArrayList<>();
    // Create list of arguments for easier processing.
    List<String> list = new ArrayList<>(args.length);
    list.addAll(Arrays.asList(args));

    // Define regular expression for grouped flag matching.
    // TODO: Escape special regex chars.
//    StringBuilder sb = new StringBuilder();
//    for (Option opt : getOptions())
//    {
//      if (!opt.requiresValue())
//        sb.append(opt.getShortName());
//    }
//    final String OPT_PATTERNF = String.format("^-([%s]+)$", sb.toString());  // Grouped flags.
    // Define regular expression for option matching.
    final String OPT_CHARS_SN = "a-z?!";
    final String OPT_CHARS_LN = "a-z?!_";
    final String OPT_PATTERN1 = String.format("^--([%s]+)(?:[=:]([^\\s]+))?$", OPT_CHARS_LN);  // Long named option.
    final String OPT_PATTERN2 = String.format("^[-/]([%s])(?:[=:]([^\\s]+))?$", OPT_CHARS_SN);  // Short named option & value.
    final String OPT_PATTERN3 = String.format("^[-/]([%s])([^\\s]+)?$", OPT_CHARS_SN);  // Short named option & value.
    final String OPT_PATTERN4 = String.format("^[-/]([%s]+)$", OPT_CHARS_SN);  // Concatenated flags.
    final Pattern optPattern1 = Pattern.compile(OPT_PATTERN1, Pattern.CASE_INSENSITIVE) ;
    final Pattern optPattern2 = Pattern.compile(OPT_PATTERN2, Pattern.CASE_INSENSITIVE) ;
    final Pattern optPattern3 = Pattern.compile(OPT_PATTERN3, Pattern.CASE_INSENSITIVE) ;
    final Pattern optPattern4 = Pattern.compile(OPT_PATTERN4, Pattern.CASE_INSENSITIVE) ;

    // Declare flag to signal end of option processing.
    boolean eoo = false;
    // Loop over arguments to process them.
    // A ListIterator is used to provide easy access to next argument.
    for (ListIterator<String> iter = list.listIterator(); iter.hasNext();)
    {
      // Get next argument to process.
      String arg = iter.next();
      if (DEBUG) System.out.printf("Argument: %s\n", arg);

      // Check for end of options, or solitary hyphen.
      if (eoo || arg.equals("-"))
      {
        spareArgs.add(arg);
        continue;
      }

      // Create regex matcher for each option pattern.
      Matcher m1 = optPattern1.matcher(arg);
      Matcher m2 = optPattern2.matcher(arg);
      Matcher m3 = optPattern3.matcher(arg);
      Matcher m4 = optPattern4.matcher(arg);

      // Check for long name match.
      if (m1.matches())
      {
        Option opt = getOptionByLongName(m1.group(1));
        if (opt == null)
          throw new OptionException(OptionException.Type.UNKNOWN_OPTION, m1.group(1));
        processOptionValue(opt, m1.group(2), iter);
      }
      // Check for end of options.
      else if (arg.startsWith("--"))
      {
        eoo = true;
        continue;  // Continue with next argument.
      }
      // Check for short name match (with delimited value).
      else if (m2.matches())
      {
        Option opt = getOptionByShortName(m2.group(1));
        if (opt == null)
          throw new OptionException(OptionException.Type.UNKNOWN_OPTION, m2.group(1));
        processOptionValue(opt, m2.group(2), iter);
      }
      // Check for short name match (with non-delimited value).
      else if (m3.matches())
      {
        Option opt = getOptionByShortName(m3.group(1));
        if (opt == null)
          throw new OptionException(OptionException.Type.UNKNOWN_OPTION, m3.group(1));
        processOptionValue(opt, m3.group(2), iter);
      }
      // Check for flag match.
      else if (m4.matches())
      {
        String optNames = m4.group(1);
        // If only a single option specified, it may take a value.
        if (optNames.length() == 1)
        {
          Option opt = getOptionByShortName(m4.group(1));
          if (opt == null)
            throw new OptionException(OptionException.Type.UNKNOWN_OPTION, m4.group(1));
          processOptionValue(opt, m4.group(2), iter);
        }
        else
        {
          for (int i = 0; i < optNames.length(); i++)
          {
            String optName = optNames.substring(i, i + 1);
            Option opt = getOptionByShortName(optName);
            if (opt == null)
              throw new OptionException(OptionException.Type.UNKNOWN_OPTION, optName);
            if (opt.requiresValue())
              throw new OptionException(OptionException.Type.NOT_FLAG, opt.getShortName(), optName); // FIXME
            if (DEBUG) System.out.printf("Setting flag option -%s\n", opt.getShortName());
            opt.addValue(Boolean.TRUE);
          }
        }
      }
      // No match to options, so add to non-option arguments.
      else
      {
        spareArgs.add(arg);
      }
    }

    // Save non-option arguments.
    nonOptionArgs = spareArgs;

    // Finally, check for value-requiring options that don't conform.
    for (Option opt : options)
    {
      int valCount = opt.getValues().size();
      if (opt.isMandatory() && opt.getValues().isEmpty())
        throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt, null);
      if (valCount < opt.getMinCount() || valCount > opt.getMaxCount())
        throw new OptionException(OptionException.Type.INVALID_OPTION_COUNT, opt, null);
    }
  }

  /**
   * Helper method for processing parsed option values.
   * For the specified {@code Option} try to assign a value from either the
   * specified string, or from the next argument from the iterator.
   */
  @SuppressWarnings("unchecked")
  private void processOptionValue(Option opt, String val, ListIterator<String> iter) throws OptionException
  {
    if (DEBUG) System.out.printf(" - processOptionValue(%s, \"%s\")\n", opt, val);
    // If value specified in this argument, process it.
    if (val != null)
    {
      if (opt.requiresValue() || opt.getType().equals(Boolean.class))
        opt.addValue(opt.parseValue(val));
      else
        throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt);
    }
    // Else value not specified, so get from next argument.
    else if (opt.requiresValue())
    {
      if (iter.hasNext())
        opt.addValue(opt.parseValue(iter.next()));
      else
        throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, opt, "null");
    }
    else if (opt.getType().equals(Boolean.class))
    {
      opt.addValue(Boolean.TRUE);
    }
  }

  /**
   * Tries to determine the likely command-line used to launch the host
   * application, and returns it.
   * @return best guess at launch command-line
   */
  private static String getLaunchCmd()
  {
    String launchCmd = null;
    // Get CLASSPATH to find if launched from JAR or CLASS.
    String cp = System.getProperty("java.class.path");
    if (Pattern.matches("^[^\\?%*:|\"<>]+\\.jar$", cp))
    {
      launchCmd = "java -jar " + cp;
    }
    else
    {
      // Find calling class with main method.
      Exception ex = new Exception();
      StackTraceElement[] st = ex.getStackTrace();
      for (int i = 2; i < st.length; i++)
      {
        if (st[i].getMethodName().equals("main"))
          launchCmd = "java " + st[i].getClassName();
      }
    }
    return (launchCmd != null) ? launchCmd : "java App";
  }

  /**
   * Prints the command-line usage message to the specified {@code PrintStream},
   * making a guess at the application launch command.
   * The usage message is automatically created using the options specified.
   * The usage can be displayed in either short or long versions, although the
   * long version uses the options' description fields, which are not required,
   * but should generally be included if intending to print the long usage.
   *
   * @param ps {@code PrintStream} to which to print usage message
   * @param longUsage whether to print long version of usage message (otherwise print short version)
   */
  public void printUsage(PrintStream ps, boolean longUsage)
  {
    printUsage(ps, longUsage, null, null);
  }

  /**
   * Prints the command-line usage message to the specified {@code PrintStream}.
   * The usage message is automatically created using the options specified.
   * The usage can be displayed in either short or long versions, although the
   * long version uses the options' description fields, which are not required,
   * but should generally be included if intending to print the long usage.
   *
   * @param ps {@code PrintStream} to which to print usage message
   * @param longUsage whether to print long version of usage message (otherwise print short version)
   * @param suffixArgs usage suffix string for defining extra arguments
   */
  public void printUsage(PrintStream ps, boolean longUsage, final String suffixArgs)
  {
    printUsage(ps, longUsage, null, suffixArgs);
  }

  /**
   * Prints the command-line usage message to the specified {@code PrintStream}.
   * The usage message is automatically created using the options specified.
   * The usage can be displayed in either short or long versions, although the
   * long version uses the options' description fields, which are not required,
   * but should generally be included if intending to print the long usage.
   *
   * @param ps {@code PrintStream} to which to print usage message
   * @param longUsage whether to print long version of usage message (otherwise print short version)
   * @param appString usage prefix string describing how application is launched; {@code null} specifies to create value automatically (e.g. &quot;java foo.bar.AppName&quot;)
   * @param suffixArgs usage suffix string for defining extra arguments
   */
  public void printUsage(PrintStream ps, boolean longUsage, final String appString, final String suffixArgs)
  {
    String appLaunch = (appString == null) ? getLaunchCmd() : appString;
    String ver = (longUsage ? "long" : "short");

    // Create buffers to hold option info.
    StringBuilder sbOpt = new StringBuilder();

    // Prepare usage formatting string for list of options.
    String fmUsageOption  = bundle.getString("option." + ver + ".mandatory");
    String fmUsageOptionO = bundle.getString("option." + ver);

    // Loop over options (in order added), formatting each one for display.
    for (Option option : options)
    {
      // Check if option has a long name.
      if (option.isHidden())
        continue;
      boolean hasLongName = option.getLongName() != null;

      // Get formatting string for the option's value-type.
      // e.g. "{0},{1} <integer>"
      String optValType = option.getType().getName();  // e.g. "java.lang.Integer"
      optValType = optValType.substring(optValType.lastIndexOf('.') + 1);  // e.g. "Integer"
      // Get type-formatting string for option (e.g. "-{0},--{1} <{2}>" for integer-type).
      String typeFormat = null;
      try { typeFormat = bundle.getString("usage.type." + optValType + (hasLongName ? "" : ".simple")); }
      catch (MissingResourceException mrx) {}
      if (typeFormat == null)
        typeFormat = bundle.getString("usage.type.unknown" + (hasLongName ? "" : ".simple"));

      // Format type string for option (e.g. "-v,--value <integer>").
      Object[] args = new String[] {
        option.getShortName(),
        option.getLongName(),
        option.getUsageTypeString()
      };
      String mUsageType = MessageFormat.format(typeFormat, args);

      // If long usage requested, create description string.
      String desc = longUsage ? (option.getDescription() != null ? option.getDescription() + "\n" : "") : "";
      // Make special case for enumerated types: prefix description with allowable values.
      if (option instanceof EnumeratedOption)
      {
        String enumDescFormat = bundle.getString("option.description.enum");
        EnumeratedOption eo = (EnumeratedOption)option;
        String enumString = eo.getAllowedValuesString();
        args = new String[] { enumString, desc };
        desc = MessageFormat.format(enumDescFormat, args);
      }
      // Make special case for LocalDate type: add example of date format.
      if (option instanceof LocalDateOption)
      {
        String descFormat = bundle.getString("option.description.LocalDate");
        args = new String[] { LocalDateOption.DF.format(LocalDate.now()), desc };
        desc = MessageFormat.format(descFormat, args);
      }

      // Format complete option usage string.
      // e.g. "
      //     -v,--value <integer>    (optional)
      //         Value to assign.
      // "
      args = new String[] { mUsageType, desc };
      String mUsageOption = null;
      if (option.isMandatory())
        mUsageOption = MessageFormat.format(fmUsageOption, args);
      else
        mUsageOption = MessageFormat.format(fmUsageOptionO, args);

      if (longUsage)
        sbOpt.append(mUsageOption);
      else
      {
        sbOpt.append(' ');  // Add space between options.
        sbOpt.append(mUsageOption);
      }
    }

    // Collate final usage string.
    String usageFormat = bundle.getString("usage." + ver);
    Object[] args = new String[] { appLaunch, sbOpt.toString(), (suffixArgs == null) ? "" : suffixArgs };
    String result = MessageFormat.format(usageFormat, args);
    ps.print(result);
  }
}
