package snaq.util.jclap;

import java.io.File;
import java.io.IOException;

/**
 * Implementation of an {@code Option} with value of type {@code String}.
 *
 * @author Giles Winstanley
 */
public class StringOption extends Option<String>
{
  /** Filter instance to use for determining whether to accept argument. */
  private Filter filter;

  /**
   * Creates a new {@code StringOption} instance.
   * @param shortName short name of the option (e.g. -n)
   * @param longName long name of the option (e.g. --name)
   * @param description helpful description of the option (printed for usage message)
   * @param minCount minimum number of occurrences required for this option
   * @param maxCount maximum number of occurrences required for this option
   */
  public StringOption(String shortName, String longName, String description,
    int minCount, int maxCount)
  {
    super(shortName, longName, description, true, minCount, maxCount);
  }

  /**
   * Creates a new {@code StringOption} instance.
   * @param shortName short name of the option (e.g. -n)
   * @param longName long name of the option (e.g. --name)
   * @param description helpful description of the option (printed for usage message)
   * @param mandatory whether this option must be specified
   * @param allowMany whether this option can take more than one value (i.e. be specified more than once)
   */
  public StringOption(String shortName, String longName, String description,
    boolean mandatory, boolean allowMany)
  {
    super(shortName, longName, description, true, mandatory, allowMany);
  }

  @Override
  protected String parseValue(String arg) throws OptionException
  {
    if (filter == null)
      return arg;
    if (filter.accept(arg))
      return arg;
    throw new OptionException(OptionException.Type.ILLEGAL_OPTION_VALUE, this, arg);
  }

  /**
   * Sets the {@code StringOption.Filter} for this instance.
   * @param filter {@code Filter} instance to use for accepting/rejecting values
   */
  public void setFilter(Filter filter) { this.filter = filter; }
  /** Returns the {@code StringOption.Filter} for this instance. */
  public Filter getFilter() { return this.filter; }

  @Override
  public Class<String> getType()
  {
    return String.class;
  }

  /**
   * Acceptance filter for {@code StringOption} instances.
   */
  public static interface Filter
  {
    /**
     * Determines whether the specified argument should be accepted for
     * use by the host {@code StringOption} instance.
     * @param arg string argument to assess for acceptance
     * @return true if the argument should be accepted, false otherwise
     */
    boolean accept(String arg);
  }

  /**
   * Implementation of an {@code StringOption.Filter} which filters based
   * on existence and type of file to which the string refers.
   */
  public static final class FileFilter implements StringOption.Filter
  {
    /** Enumeration of possible file existence states to accept. */
    public static enum AcceptExistance
    {
      /** Use to accept only existing files. */
      ACCEPT_EXISTING,
      /** Use to accept only non-existing files. */
      ACCEPT_NON_EXISTING,
      /** Use to accept existing or non-existing files. */
      ACCEPT_ALL
    }
    /** Enumeration of possible types to accept. */
    public static enum AcceptFileType
    {
      /** Use to accept only files (not directories). */
      ACCEPT_FILE,
      /** Use to accept only directories (not files). */
      ACCEPT_DIR,
      /** Use to accept only files or directories. */
      ACCEPT_ALL
    }
    /** Determines acceptance of file based on existance of file. */
    private AcceptExistance eType = AcceptExistance.ACCEPT_ALL;
    /** Determines the file types to accept. */
    private AcceptFileType fType = AcceptFileType.ACCEPT_ALL;

    /**
     * Creates a new {@code FileOption} instance (allows files/folders).
     * @param eType type of files to accept/reject based on existence
     * @param fType type of files to accept/reject based on file/directory status
     */
    public FileFilter(AcceptExistance eType, AcceptFileType fType)
    {
      this.eType = eType;
      this.fType = fType;
    }

    /**
     * Determines whether the string argument is valid as an option value.
     * @param arg string argument to check for validity as an option value
     * @return Whether the string argument is valid as an option value.
     */
    @Override
    public boolean accept(String arg)
    {
      try
      {
        File f = new File(arg).getCanonicalFile();
//        if (DEBUG) System.err.printf("FileFilter:[%s,%s] arg:%s file:%s exists=%b isDir=%b\n", eType, fType, arg, f.getAbsolutePath(), f.exists(), f.isDirectory());
        switch (eType)
        {
          case ACCEPT_NON_EXISTING:
            if (f.exists())
              return false;
            break;
          case ACCEPT_EXISTING:
            if (!f.exists())
              return false;
            break;
          case ACCEPT_ALL:
          default:
        }
        switch (fType)
        {
          case ACCEPT_DIR:
            if (!f.isDirectory())
              return false;
            break;
          case ACCEPT_FILE:
            if (!f.isFile())
              return false;
            break;
          // This case performs checks to ensure File instance actually
          // denotes a real system-level file, not just garbage.
          case ACCEPT_ALL:
            if (f.isFile() || f.isDirectory())
              return true;
            break;
          default:
        }
        return true;
      }
      catch (IOException iox)
      {
//        if (DEBUG) iox.printStackTrace();
        return false;
      }
    }
  }
}
