/*
 * The SmartWeb Framework
 * Copyright (C) 2004-2006
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * For further informations on the SmartWeb Framework please visit
 *
 *                        http://smartweb.sourceforge.net
 */
package net.smartlab.web.auth;

import java.util.StringTokenizer;

import javax.servlet.jsp.JspException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * This JSP tag is used to check the validity of the current user both in terms
 * of authentication and authorization. In its basic usage the tag checks if a
 * user is associated to the current session. If the optional <code>role</code>
 * attribute is specified it identifies a comma separated list of roles to which
 * the user <b>may </b> be associated to pass the validity check. In addition a
 * context could be optionally specified using the <code>name</code> attribute
 * causing the validity check to be performed on the permessions allowed on the
 * <code>SecureBusinessObject</code> binded with that name.
 * 
 * @author rlogiacco
 * @author svetrini
 * @jsp.tag name = "isValid" description="checks if a user was previously
 *          authenticated or if it's authorized by a role" body-content="JSP"
 */
public class ValidTag extends org.apache.struts.taglib.logic.PresentTag {
	
	/**
	 * Provides logging for the test cases.
	 * @uml.property  name="logger"
	 * @uml.associationEnd  multiplicity="(1 1)"
	 */
	protected Log logger = LogFactory.getLog(this.getClass());

	/* # net.smartlab.web.auth.User Dependency_Link1 */
	private static final long serialVersionUID = -3426773639709918540L;

	/**
	 * The roles the user <b>may </b> be associated to pass the validity check.
	 * @uml.property  name="roles"
	 * @uml.associationEnd  multiplicity="(0 -1)"
	 */
	private Role[] roles;
	/**
	 * @uml.property  name="functions"
	 * @uml.associationEnd  multiplicity="(0 -1)"
	 */
	private Privilege[] functions;

	/**
	 * @uml.property  name="name"
	 */
	private String name;


	/**
	 * Default empty constructor.
	 */
	public ValidTag() {
		super.name = UserAction.SESSION_KEY;
		super.scope = "session";
	}

	/**
	 * @see org.apache.struts.taglib.logic.ConditionalTagBase#condition()
	 */
	protected boolean condition() throws JspException {
		return (condition(true));
	}

	/**
	 * @see org.apache.struts.taglib.logic.PresentTag#condition(boolean)
	 */
	public boolean condition(boolean desired) throws JspException {
		return check(desired, super.condition(desired), functions, roles, name);
	}

	/**
	 * @param desired
	 * @param superConition
	 * @param locFunctions
	 * @param locRoles
	 * @param locName
	 * @return
	 * @throws JspException
	 */
	public boolean check(boolean desired, boolean superConition,
			Privilege[] locFunctions, Role[] locRoles, String locName){
		if (logger.isDebugEnabled()){
			logger.debug("desired: "+desired);
			logger.debug("superConition: "+desired);
			logger.debug("locName: "+locName);
			if (locRoles!=null)
			for (int i = 0; i < locRoles.length; i++) {
				Role role = locRoles[i];
				logger.debug("locRoles["+i+"]: "+role.getDisplay());				
			}
			if (locFunctions!=null)
			for (int i = 0; i < locFunctions.length; i++) {
				Privilege privilege = locFunctions[i];
				logger.debug("locFunctions["+i+"]: "+privilege);				
			}
		}

		User user = User.get();	
		logger.debug("current user: "+User.get().getDisplay());
		if (User.isGuest(user)) return !desired;
		if ( ( locRoles != null || locFunctions!=null )&& superConition) {
			// a user was authenticated and roles were specified
			
			
			if (locName != null) {
				// a scope was specified
				Scope scope = new Scope();
				scope.setTypeId(locName);
					if (locRoles != null ){
						
						for (int i = 0; i < locRoles.length; i++) {
							boolean isInRole = user.isInRole(locRoles[i]);
							logger.debug("check :: user ( "+user.getDisplay()+" ) isInRole( "+locRoles[i]+" ) = "+isInRole);							
							if (isInRole) {								
								return desired;
							}
						}
					}
					if (locFunctions != null ){
						
						for (int i = 0; i < locFunctions.length; i++) {
							boolean hasPrivilege = user.hasPrivilege(locFunctions[i],scope);
							logger.debug("check :: user ("+user.getDisplay()+") hasPrivilege( "+locFunctions[i]+","+scope+" ) = "+hasPrivilege);
							if (!hasPrivilege) {								
								return !desired;
							}
						}
						return desired;
					}
				
				
			} else {
				// no scope specified
				if (locRoles != null ){
					
					for (int i = 0; i < locRoles.length; i++) {
						boolean isInRole = user.isInRole(locRoles[i]);
						logger.debug("check :: user ( "+user.getDisplay()+" ) isInRole( "+locRoles[i]+" ) = "+isInRole);							
						if (isInRole) {								
							return desired;
						}
					}
				}
				if (locFunctions != null ){
					
					for (int i = 0; i < locFunctions.length; i++) {
						boolean hasPrivilege = user.hasPrivilege(locFunctions[i]);
						logger.debug("check :: user ("+user.getDisplay()+") hasPrivilege( "+locFunctions[i]+" ) = "+hasPrivilege);
						if (!hasPrivilege) {								
							return !desired;
						}
					}
					return desired;
				}
				
			}
			return !desired;
		}
		return superConition;
	}

	/**
	 * @see org.apache.struts.taglib.logic.ConditionalTagBase#setRole(java.lang.String)
	 * @jsp.attribute required="false" rtexprvalue="true" type="string"
	 *                description="a comma separated list of authorization
	 *                roles"
	 */
	public void setRole(String role) {
		StringTokenizer tokenizer = new StringTokenizer(role, ",");
		this.roles = new Role[tokenizer.countTokens()];
		for (int i = 0; i < roles.length; i++) {
			roles[i] = new Role(tokenizer.nextToken().trim());
		}
	}
	
	public void setFunction(String function) {
		StringTokenizer tokenizer = new StringTokenizer(function, ",");
		this.functions = new Privilege[tokenizer.countTokens()];
		for (int i = 0; i < functions.length; i++) {
			functions[i] = new Privilege("view",tokenizer.nextToken().trim(),"r");//
		}
	}

	/**
	 * @see org.apache.struts.taglib.logic.ConditionalTagBase#setScope(java.lang.String)
	 * @jsp.attribute required="false" rtexprvalue="true" type="string"
	 *                description="the name of the attribute representing the
	 *                context against which the authorization must be performed"
	 */
	public void setScope(String scope) {
		this.name = scope;
	}
}
