/*
 * The SmartWeb Framework
 * Copyright (C) 2004-2006
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * For further informations on the SmartWeb Framework please visit
 *
 *                        http://smartweb.sourceforge.net
 */
package net.smartlab.web.auth;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import net.smartlab.web.BusinessObjectFactory;
import net.smartlab.web.DAOException;

import org.hibernate.HibernateException;
import org.hibernate.Query;
import org.hibernate.Session;

/**
 * @author rlogiacco
 * @author svetrini
 */
public class UserFactory extends BusinessObjectFactory {

	/**
	 * Stores the singleton <code>instance</code> for this class.
	 * @uml.property  name="instance"
	 */
	private static UserFactory instance;


	/**
	 * Blocks class instantiation to enforce the singleton pattern.
	 */
	private UserFactory() {
		super();
	}

	/**
	 * @return Returns the instance.
	 * @uml.property name="instance"
	 */
	public static synchronized UserFactory getInstance() {
		if (instance == null) {
			instance = new UserFactory();
		}
		return instance;
	}

	/**
	 * @see net.smartlab.web.BusinessObjectFactory#getMappedClass()
	 */
	public Class getMappedClass() {
		return User.class;
	}

	public User findByUsername(String username) throws DAOException {
		if (logger.isDebugEnabled()) {
			logger.debug("findByUsername(username = " + username + ") - start");
		}
		try {
			Session session = super.current();
			Query query = session.createQuery("select u from User as u where u.username=:username");
			query.setString("username", username);
			User user = (User)query.uniqueResult();

			if (user != null) {
				if (logger.isTraceEnabled()) {
					logger.trace("   matching user found");
				}
				//FIXME
				//user.getPolicy();
				return user;
			} else {
				if (logger.isTraceEnabled()) {
					logger.trace("   invalid user");
				}
				return null;
			}
		} catch (Exception he) {
			logger.error("findByUsername(username = " + username + ") - error", he);
			throw new DAOException(he);
		}
	}

	/**
	 * Returns the number of users currently having the specified status.
	 * 
	 * @param status the status of the users to count.
	 * @return the number of users currently having the specified status.
	 * @throws DAOException if something unusual happens accessing the
	 *         underlying data store.
	 */
	public long count(User.Status status) throws DAOException {
		if (logger.isDebugEnabled()) {
			logger.debug("count(User.Status) - start");
		}
		try {
			Query query = super.current().createQuery("select count(*) from User where status=:status");
			query.setInteger("status", status.getId());
			return ((Long)query.uniqueResult()).longValue();
		} catch (HibernateException he) {
			logger.error("count(User.Status) - failed", he);
			throw new DAOException(he);
		}
	}

	/**
	 * Returns the number of users currently in each status.
	 * 
	 * @return the number of users currently in each status.
	 * @throws DAOException
	 */
	public Map count() throws DAOException {
		if (logger.isDebugEnabled()) {
			logger.debug("count() - start");
		}
		try {
			Map counts = new HashMap();
			Long zero = new Long(0);
			counts.put(User.Status.ENABLED, zero);
			counts.put(User.Status.DISABLED, zero);
			counts.put(User.Status.ONLINE, zero);
			counts.put(User.Status.PENDING, zero);
			Query query = super.current().createQuery("select status,count(*) from User group by status");
			Iterator results = query.iterate();
			while (results.hasNext()) {
				Object[] row = (Object[])results.next();
				counts.put(row[0], row[1]);
			}
			return counts;
		} catch (HibernateException he) {
			logger.error("count() - failed", he);
			throw new DAOException(he);
		}
	}

}
