/*
 * The SmartWeb Framework
 * Copyright (C) 2004-2006
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * For further informations on the SmartWeb Framework please visit
 *
 *                        http://smartweb.sourceforge.net
 */
package net.smartlab.web.auth;

import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import net.smartlab.web.DAOException;
import net.smartlab.web.Enumeration;

import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.lang.builder.ToStringBuilder;

/**
 * This class represents the minimum set of informations needed to authenticate,
 * authorize, audit and manage a person or system.
 * 
 * @author rlogiacco
 * @author svetrini
 * @author pmoretti
 * 
 * @hibernate.joined-subclass schema="auth" table="`user`"
 * @hibernate.joined-subclass-key column="`id`"
 */
public class User extends Subject {

	private final static long serialVersionUID = 1921921080970903250L;

	/**
	 * Carries the current user between tiers in the same JVM.
	 */
	private final static ThreadLocal current = new ThreadLocal();

	/**
	 * The list of user statuses under administrative control: it doesn't include statuses automatically managed like on-line, off-line and so on.
	 * @uml.property  name="STATUS_LIST"
	 * @uml.associationEnd  multiplicity="(0 -1)"
	 */
	public final static Status[] STATUS_LIST = new Status[] {Status.PENDING, Status.ENABLED, Status.DISABLED};

	/**
	 * Instance representing a non authenticated access.
	 */
	public final static User GUEST = new User();
	static {
		GUEST.id = -1;
		GUEST.display = "Guest";
		GUEST.securityToken = "GuestSecurityToken";
		try {
			Permission permission = new Permission();
			Role role = (Role)RoleFactory.getInstance().findByKey("guest");
			permission.setRoleId(role.getId());
			permission.setSubject(GUEST);
//			Scope any = new Scope();
//			any.setTypeId(Scope.ANY);
//			permission.setScope(any);
			GUEST.permissions.add(permission);
		} catch (Exception e) {
			GUEST.logger.info("No permission granted to GUEST user");
		}
	}

	/**
	 * The user credentials.
	 * @uml.property  name="credentials"
	 * @uml.associationEnd  multiplicity="(1 1)" inverse="user:net.smartlab.web.auth.Credentials"
	 */
	private Credentials credentials = new Credentials();

	/**
	 * The user status.
	 * @uml.property  name="status"
	 * @uml.associationEnd  
	 */
	private Status status = Status.PENDING;

	/**
	 * Contains any <code>Permission</code> granted to this user both directly
	 * or inherited by the groups it belongs to.
	 * 
	 * @uml.property name="policy"
	 */
	//private Set policy = new HashSet();

	/**
	 * Contains the groups this user belongs to.
	 * @link   aggregation <{Group}>
	 * @directed   directed
	 * @supplierCardinality   0..*
	 * @uml.property  name="groups"
	 */
	private Set groups = new HashSet();

	/**
	 * TODO documentation
	 * @uml.property  name="features"
	 */
	private Map features;

	/**
	 * TODO documentation
	 * @uml.property  name="securityToken"
	 */
	private String securityToken;

	/**
	 * TODO documentation
	 * @uml.property  name="created"
	 */
	private Date created = new Date();

	/**
	 * TODO documentation
	 * @uml.property  name="lastAccess"
	 */
	private Date lastAccess;


	/**
	 * @see net.smartlab.web.auth.Subject#getId()
	 * @hibernate.id column="`id`" generator-class="native"
	 */
	public long getId() {
		return super.getId();
	}

	/**
	 * TODO documentation
	 * 
	 * @param privilege
	 * @return
	 */
	public boolean hasPrivilege(Privilege privilege) {
		return hasPrivilege(privilege, null);
	}

	/**
	 * TODO documentation
	 * 
	 * @param privilege
	 * @param scope
	 * @return
	 */
	public boolean hasPrivilege(Privilege privilege, Scope scope) {
		Set permissions = this.getPolicy();
		for (Iterator iterator = permissions.iterator(); iterator.hasNext();) {
			Permission perm = (Permission)iterator.next();
			Set privileges = Collections.EMPTY_SET;
			try {
				Role userRole = (Role)RoleFactory.getInstance().findByKey(perm.getRoleId());
				privileges = userRole.getPrivileges();
			} catch (DAOException e) {
				// TODO
			}
			for (Iterator iterator2 = privileges.iterator(); iterator2.hasNext();) {
				Privilege priv = (Privilege)iterator2.next();
				if (scope == null) {
					if (privilege.match(priv))
						return true;
				} else {
					if (privilege.match(priv) && scope.match(perm.getScope()))
						return true;
				}
			}
		}
		return false;
	}

	/**
	 * Returns the unique name used for authentication.
	 * 
	 * @return the unique name used for authentication.
	 * @hibernate.property column="`username`" length="50" not-null="true"
	 *                     unique="true"
	 */
	public String getUsername() {
		return this.credentials.getUsername();
	}

	/**
	 * Sets the unique name used for authentication.
	 * 
	 * @param username the unique name used for authentication.
	 */
	public void setUsername(String username) {
		this.credentials.setUsername(username);
	}

	/**
	 * @return
	 */
	public String getPassword() {
		if (this.credentials.getSecret() == null ) return "";
		return new String(this.credentials.getSecret());
	}

	/**
	 * @param password
	 */
	public void setPassword(String password) {
		this.credentials.setSecret(password.getBytes());
	}

	/**
	 * Returns the credentials used for authentication.
	 * 
	 * @return the credentials used for authentication.
	 * @hibernate.property column="`secret`" length="255"
	 */
	public byte[] getSecret() {
		return this.credentials.getSecret();
	}

	/**
	 * TODO documentation
	 * 
	 * @param secret
	 */
	public void setSecret(byte[] secret) {
		this.credentials.setSecret(secret);
	}

	/**
	 * Returns the groups this user belongs to.
	 * 
	 * @return the groups this user belongs to.
	 * @hibernate.set lazy="false" schema="auth" table="`group_user`"
	 *                cascade="none"
	 * @hibernate.collection-key column="`user`"
	 * @hibernate.collection-many-to-many column="`group`"
	 *                                    class="net.smartlab.web.auth.Group"
	 * @uml.property name="groups"
	 */
	public Set getGroups() {
		return groups;
	}

	/**
	 * Sets the groups this user belongs to.
	 * 
	 * @param groups the groups this user belongs to.
	 * @uml.property name="groups"
	 */
	protected void setGroups(Set groups) {
		this.groups = groups;
	}

	/**
	 * Empties the user group set ensuring this user doesn't belong to any
	 * group.
	 */
	public void clearGroups() {
		groups.clear();
	}

	/**
	 * Adds a group to the set this user belongs to.
	 * 
	 * @param group the group to add to the set this user belongs to.
	 */
	public void add(Group group) {
		groups.add(group);
	}

	/**
	 * Removes a group from the set this user belongs to.
	 * 
	 * @param group the group to remove from the set this user belongs to.
	 */
	public void remove(Group group) {
		groups.remove(group);
	}

	/**
	 * Return all the permissions granted to this user, both the ones granted
	 * directly and the ones it has grants inherited from the groups it belongs
	 * to.
	 * 
	 * @return all the permissions this user is authorized to impersonate.
	 * @FIXME hibernate.property insert="false" update="false" formula="( select User from `group_user` where `group_user`.group=id )"
	 * @uml.property name="policy"
	 */
	public Set getPolicy() {
		// FIXME calculate policy in UserFactory
		try {
			Set result = PermissionFactory.getInstance().listPolicy(""+this.getId());
			result.addAll(GUEST.permissions);
			return result;
		} catch (DAOException e) {
			logger.error(e);
			return null;
		}
		// return policy;
//		if (this.policy.size() == 0) {
//			HashSet permissions = new HashSet();
//			permissions.addAll(this.getPermissions());
//			for (Iterator iterator = this.groups.iterator(); iterator.hasNext();) {
//				Group group = (Group)iterator.next();
//				permissions.addAll(group.getPermissions());
//			}
//			this.policy = permissions;
//		}
//		return this.policy;
	}

	/**
	 * Sets all the permissions this user is authorized to impersonate.
	 * 
	 * @param policy all the permission this user is authorized to impersonate.
	 * @uml.property name="policy"
	 */
	protected void setPolicy(Set policy) {
		//this.policy = policy;
	}

	/**
	 * TODO documentation
	 * @return
	 * @uml.property  name="features"
	 */
	public Map getFeatures() {
		return features;
	}

	/**
	 * TODO documentation
	 * @param  features
	 * @uml.property  name="features"
	 */
	protected void setFeatures(Map features) {
		this.features = features;
	}

	/**
	 * Return the user status.
	 * 
	 * @return the user status.
	 * @hibernate.property column="`status`"
	 *                     type="net.smartlab.web.auth.User$Status"
	 * @uml.property name="status"
	 */
	public Status getStatus() {
		return status;
	}

	/**
	 * Returns the list of statuses that can be setted on this user.
	 * 
	 * @return the list of statuses that can be setted on this user.
	 */
	public Status[] getStatuses() {
		return STATUS_LIST;
	}

	/**
	 * Sets the user status.
	 * 
	 * @param status the user status.
	 * @uml.property name="status"
	 */
	public void setStatus(Status status) {
		this.status = status;
	}

	/**
	 * TODO documentation
	 * 
	 * @return
	 * @uml.property name="securityToken"
	 */
	public String getSecurityToken() {
		return securityToken;
	}

	/**
	 * TODO documentation
	 * 
	 * @param securityToken
	 * @uml.property name="securityToken"
	 */
	protected void setSecurityToken(String securityToken) {
		this.securityToken = securityToken;
	}

	/**
	 * Returns the created.
	 * 
	 * @return the created.
	 * @uml.property name="created"
	 */
	public Date getCreated() {
		return created;
	}

	/**
	 * Returns the lastAccess.
	 * 
	 * @return the lastAccess.
	 * @uml.property name="lastAccess"
	 */
	public Date getLastAccess() {
		return lastAccess;
	}

	/**
	 * Logs in the user and sets the status accordingly. This method doesn't
	 * verify if the user was previously logged in so allows multiple logins
	 * with the same credentials.
	 * 
	 * @throws AuthenticationException if the user user status doesn't allow the
	 *         log in operation.
	 */
	public void login() throws AuthenticationException {
		if (logger.isDebugEnabled()) {
			logger.debug("user - login() - start");
		}
		if (this.status == Status.DISABLED || this.status == Status.PENDING) {
			throw new AuthenticationException("errors.activeness");
		} else {
			this.status = Status.ONLINE;
			this.lastAccess = new Date();
		}
		logger.debug("before user set - user: "+ User.get().getDisplay());
		User.set(this);
		logger.debug("after user set - user: "+ User.get().getDisplay());
	}

	/**
	 * Logs the user out and sets the user status accordingly.
	 * 
	 * @throws AuthenticationException if the current user status doesn't allow
	 *         the logout operation.
	 */
	public void logout() throws AuthenticationException {
		if (logger.isDebugEnabled()) {
			logger.debug("logout() - start");
		}
		if (this.status != Status.PENDING && this.status != Status.DISABLED) {
			this.status = Status.ENABLED;
		} else {
			throw new AuthenticationException("errors.activeness");
		}
		logger.debug("before user set - user: "+ User.get().getDisplay());
		User.set(null);
		logger.debug("after user set - user: "+ User.get().getDisplay());
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return new ToStringBuilder(this).appendToString(super.toString()).append("username", credentials.getUsername())
				.append("status", status).toString();
	}


	/**
	 * The status of a <code>User</code>. Instances of this class identify the actual status of the user lifecycle.
	 * @author   rlogiacco@users.sourceforge.net
	 */
	public static class Status extends Enumeration {

		private final static long serialVersionUID = -1647801550717471996L;

		/**
		 * Users in the <b>pending </b> status were registered but are not yet
		 * authorized to access the system. While in this status users are
		 * considered freezed and awaiting for a change.
		 */
		public final static Status PENDING = new Status('P', "user.status.pending");

		/**
		 * Users in the <b>enabled </b> status are authorized to access the
		 * system. While in this status users are considered active but not
		 * connected.
		 */
		public final static Status ENABLED = new Status('E', "user.status.enabled");

		/**
		 * Users in the <b>disabled </b> status are no more authorized to access
		 * the system. Once entered in this status users are considered blocked
		 * and available for clean up.
		 */
		public final static Status DISABLED = new Status('D', "user.status.disabled");

		/**
		 * Users in the <b>online</b> status are actually logged into the
		 * system. While in this status users are considered active and
		 * connected.
		 */
		public final static Status ONLINE = new Status('O', "user.status.online");

		/**
		 * Users in the <b>generated</b> status are fully capable to log into
		 * the system but need some sort of maintenance. Usually users fall in
		 * this status because the secret has been forgotten or an intermediate
		 * step is needed after account activation.
		 */
		public final static Status GENERATED = new Status('G', "user.status.generated");


		/**
		 * Default empty constructor.
		 */
		public Status() {
			super();
		}

		/**
		 * Creates a valid instance of Status.
		 * 
		 * @param id the unique identifier.
		 * @param display the display name associated to the status.
		 */
		public Status(int id, String display) {
			super(id, display);
		}

		/**
		 * @see net.smartlab.web.Enumeration#decode(int)
		 */
		public Enumeration decode(int id) {
			switch (id) {
				case 'D':
					return DISABLED;
				case 'P':
					return PENDING;
				case 'E':
					return ENABLED;
				case 'O':
					return ONLINE;
				case 'G':
					return GENERATED;
				default:
					return PENDING;
			}
		}
	}


	/**
	 * TODO documentation
	 * 
	 * @param user
	 */
	protected static void set(User user) {
		User.current.set(user);
	}

	/**
	 * TODO documentation
	 * 
	 * @return user
	 */
	public static User get() {
		User result = null;
		result = (User)User.current.get();
		if (result == null) result = GUEST;
		return result;
	}

	/**
	 * TODO documentation
	 * 
	 * @return
	 */
	public static boolean isGuest() {
		return User.isGuest(User.get());
	}

	/**
	 * TODO documentation
	 * 
	 * @param user
	 * @return
	 */
	public static boolean isGuest(User user) {
		return (user == null || user == GUEST || user.id == -1);
	}

	public boolean isInRole(Role role) {
		Set permissions = this.getPolicy();
		for (Iterator iterator = permissions.iterator(); iterator.hasNext();) {
			Permission perm = (Permission)iterator.next();
			if (perm.getRoleId().equals(role.getId()))
				return true;
		}
		return false;
	}

	public boolean isStrictedInRole(Role role) {
		Set permissions = this.getPermissions();
		for (Iterator iterator = permissions.iterator(); iterator.hasNext();) {
			Permission perm = (Permission)iterator.next();
			if (perm.getRoleId().equals(role.getId()))
				return true;
		}
		return false;
	}

	/**
	 * @return
	 * @uml.property  name="credentials"
	 */
	public Credentials getCredentials() {
		return credentials;
	}

	/**
	 * @param group
	 * @return
	 */
	public boolean isInGroup(Group group) {
		return this.getGroups().contains(group);
	}

	/**
	 * @see java.lang.Object#equals(Object)
	 */
	public boolean equals(Object object) {
		if (!(object instanceof User)) {
			return false;
		}
		User rhs = (User)object;
		return new EqualsBuilder().append(this.getUsername(), rhs.getUsername()).isEquals();
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return new HashCodeBuilder(1209070065, -1124530119).append(this.getUsername()).toHashCode();
	}
}
