/*
 * The SmartWeb Framework
 * Copyright (C) 2004-2006
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * For further informations on the SmartWeb Framework please visit
 *
 *                        http://smartweb.sourceforge.net
 */
package net.smartlab.web.auth;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import net.smartlab.web.BusinessObject;
import org.apache.commons.lang.builder.ToStringBuilder;

/**
 * This abstract class is a common ancestor for <code>User</code> and
 * <code>Group</code>.
 * 
 * @author svetrini
 * @author rlogiacco
 * @hibernate.class abstract="true" schema="auth" table="`subject`"
 */
public abstract class Subject extends BusinessObject {

	/**
	 * The unique identifier.
	 * @uml.property  name="id"
	 */
	protected long id;

	/**
	 * The string to be shown to the end user.
	 * @uml.property  name="display"
	 */
	protected String display;

	/**
	 * An associative map of properties related to this subject. This map should be used to enrich the subject with custom values. Be carefull as search on a specific combination of properties <b>is not supported </b> by default as it could be really expensive.
	 * @uml.property  name="properties"
	 */
	protected Map properties = new HashMap();

	/**
	 * Contains the <code>Permission</code>s associated to this subject.
	 * @uml.property  name="permissions"
	 * @uml.associationEnd  multiplicity="(0 -1)" aggregation="composite" inverse="subject:net.smartlab.web.auth.Permission"
	 */
	protected Set permissions = new HashSet();

	/**
	 * Reference to an <code>Handler</code> which constraints <b>any</b> activity of this subject.
	 * @uml.property  name="handler"
	 * @uml.associationEnd  inverse="subject:net.smartlab.web.auth.Handler"
	 */
	protected Handler handler;


	/**
	 * Returns the unique identification number.
	 * 
	 * @return the unique identification number.
	 * @hibernate.id column="`id`" unsaved-value="0" generator-class="native"
	 * @uml.property name="id"
	 */
	public long getId() {
		return id;
	}

	/**
	 * Sets the unique identification number.
	 * 
	 * @param id the unique identification number.
	 * @uml.property name="id"
	 */
	protected void setId(long id) {
		this.id = id;
	}

	/**
	 * Returns the display name.
	 * 
	 * @return the display name.
	 * @hibernate.property column="`display`" length="50" not-null="true"
	 * @uml.property name="display"
	 */
	public String getDisplay() {
		return display;
	}

	/**
	 * Sets the display name.
	 * 
	 * @param display the display name.
	 * @uml.property name="display"
	 */
	public void setDisplay(String display) {
		this.display = display;
	}

	/**
	 * Returns an associative map of properties related to this subject. This
	 * map should be used to enrich the principal with custom values. Be careful
	 * as search on a specific combination of properties <b>is not supported
	 * </b> by default as it could be really expensive.
 	 *
     * @hibernate.map lazy="false" outer-join="true" schema="auth"
     *                table="`property`" inverse="false" cascade="all"
     * @hibernate.collection-key column="`subject`"
     * @hibernate.collection-index column="`name`" type="java.lang.String"
     *                             length="50"
     * @hibernate.collection-element column="`value`" type="java.lang.String"
     *                               length="255"
	 * @return
	 * @uml.property name="properties"
	 */
	public Map getProperties() {
		return properties;
	}

	/**
	 * Sets an associative map of properties related to this entry. This map
	 * should be used to enrich the entry with custom values. Be carefull as
	 * search on a specific combination of properties <b>is not supported </b>
	 * by default as it could be really expensive.
	 * 
	 * @param properties the properties to associate to this entry.
	 * @uml.property name="properties"
	 */
	public void setProperties(Map properties) {
		this.properties = properties;
	}

	/**
	 * Sets a property into the associative map of properties related to this
	 * entry. This map should be used to enrich the entry with custom values. Be
	 * careful as search on a specific combination of properties <b>is not
	 * supported </b> by default as it could be really expensive.
	 * 
	 * @param name the property name.
	 * @param value the property value.
	 */
	public void setProperty(String name, String value) {
		properties.put(name, value);
	}

	/**
	 * Removes a property from the associative map of properties related to this
	 * entry. This map should be used to enrich the entry with custom values. Be
	 * careful as search on a specific combination of properties <b>is not
	 * supported</b> by default as it could be really expensive.
	 * 
	 * @param name the property name.
	 */
	public void removeProperty(String name) {
		properties.remove(name);
	}

	/**
	 * Finds a property into the associative map of properties related to this
	 * entry. This map should be used to enrich the entry with custom values. Be
	 * careful as search on a specific combination of properties <b>is not
	 * supported </b> by default as it could be really expensive.
	 * 
	 * @param name the property name.
	 * @return the property value or <code>null</code> if a property with that
	 *         name was never defined.
	 */
	public String getProperty(String name) {
		return (String)properties.get(name);
	}

	/**
	 * Returns the permissions this subject is associated to.
	 * 
	 * @hibernate.set lazy="false" schema="auth" table="`permission`"
	 *                cascade="none"
	 * @hibernate.collection-key column="`subject`"
	 * @hibernate.collection-one-to-many column="`permission`" class="net.smartlab.web.auth.Permission"
	 * @return the permissions this user is associated to.
	 * @uml.property name="permissions"
	 */
	public Set getPermissions() {
		return permissions;
	}

	/**
	 * Sets the permissions this user is associated to.
	 * 
	 * @param permissions the permissions this user is associated to.
	 * @uml.property name="permissions"
	 */
	public void setPermissions(Set permissions) {
		logger.debug("adding permissions: "+permissions);
		this.permissions = permissions;
	}

	/**
	 * Removes all of the elements from this collection (optional operation).
	 * 
	 * @see java.util.Collection#clear()
	 * @uml.property name="permissions"
	 */
	public void clearPermissions() {
		permissions.clear();
	}

	/**
	 * Adds a permission to the set this user is associated to.
	 * 
	 * @param permission the permission to add to the set this user is
	 *        associated to.
	 */
	public void add(Permission permission) {
		permissions.add(permission);
	}

	/**
	 * Removes a permission from the set this user is associated to.
	 * 
	 * @param permission the permission to remove from the set this user is
	 *        associated to.
	 */
	public void remove(Permission permission) {
		permissions.remove(permission);
	}

	/**
	 * Returns the handler.
	 * 
	 * @return returns the handler.
	 * @uml.property name="handler"
	 */
	public Handler getHandler() {
		return handler;
	}

	/**
	 * Sets the handler.
	 * 
	 * @param handler the handler to set.
	 * @uml.property name="handler"
	 */
	public void setHandler(Handler handler) {
		this.handler = handler;
	}
	
	/**
	 * @see net.smartlab.web.BusinessObject#getVersion()
	 * @hibernate.version
	 */
	public long getVersion() {
		return super.getVersion();
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return new ToStringBuilder(this).append("id", id).append("display", display).append("properties", properties)
				.append("handler", handler).appendToString(super.toString()).toString();
	}


	
	
}
