/*
 * The SmartWeb Framework
 * Copyright (C) 2004-2006
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * For further informations on the SmartWeb Framework please visit
 *
 *                        http://smartweb.sourceforge.net
 */
package net.smartlab.web.auth;

import java.util.HashSet;
import java.util.Set;
import org.apache.commons.lang.builder.ToStringBuilder;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;

/**
 * This class aggregates users to simplify their management.
 * 
 * @author rlogiacco
 * @author svetrini
 * 
 * @hibernate.joined-subclass schema="auth" table="`group`"
 * @hibernate.joined-subclass-key column="`id`"
 */
public class Group extends Subject {

	private final static long serialVersionUID = -6251419333679466935L;

	/**
	 * TODO documentation
	 * @uml.property  name="description"
	 */
	private String description;

	/**
	 * Count of users belonging to this group.
	 * @uml.property  name="size"
	 */
	private int size;

	/**
	 * The users belonging to this group.
	 * @link   aggregation <{User}>
	 * @directed   directed
	 * @supplierCardinality   0..*
	 * @uml.property  name="users"
	 * @uml.associationEnd  multiplicity="(0 -1)" aggregation="composite" inverse="groups:net.smartlab.web.auth.User"
	 */
	private Set users = new HashSet();

	/**
	 * @see net.smartlab.web.auth.Subject#getId()
	 * @hibernate.id column="`id`" generator-class="native"
	 */
	public long getId() {
		return super.getId();
	}

	/**
	 * Returns the description.
	 * @return  the description
	 * @hibernate.property  column="`description`" type="text"
	 * @uml.property  name="description"
	 */
	public String getDescription() {
		return description;
	}

	/**
	 * Sets the description.
	 * @param description  the description to set
	 * @uml.property  name="description"
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	/**
	 * Returns the users belonging to this group.
	 * 
	 * @return the users belonging to this group.
	 * @hibernate.set lazy="false" schema="auth" table="`group_user`"
	 *                cascade="none" inverse="true"
	 * @hibernate.collection-key column="`group`"
	 * @hibernate.collection-many-to-many column="`user`"
	 *                                    class="net.smartlab.web.auth.User"
	 * @uml.property name="users"
	 */
	public Set getUsers() {
		return users;
	}

	/**
	 * Sets the users belonging to this group.
	 * 
	 * @param users
	 *            the users to set.
	 * @uml.property name="users"
	 */
	public void setUsers(Set users) {
		this.users = users;
	}

	/**
	 * Removes any user from this group.
	 */
	public void clearUsers() {
		users.clear();
		size = 0;
	}

	/**
	 * Adds a user to this group.
	 * 
	 * @param user
	 *            the user to add.
	 */
	public void add(User user) {
		users.add(user);
	}

	/**
	 * Removes a user from this group.
	 * 
	 * @param user
	 *            the user to remove.
	 */
	public void remove(User user) {
		users.remove(user);
	}

	public boolean hasUser(User user){
		return users.contains(user);
	}
	/**
	 * Returns the number of users belonging to this group.
	 * 
	 * @return the number of users belonging to this group.
	 * @FIXME hibernate.property insert="false" update="false" formula="( select count(*) from group_user where group_user.group=id )"
	 * @uml.property name="size"
	 */
	public int getSize() {
		if (this.getUsers()!=null){
			this.size = this.getUsers().size(); 
		}
		return this.size;	 
		
	}

	/**
	 * Sets the number of users belonging to this group.
	 * 
	 * @param size
	 *            the number of users belonging to this group.
	 * @uml.property name="size"
	 */
	protected void setSize(int size) {
		this.size = size;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return new ToStringBuilder(this).append(super.toString()).append(
				"size", size).toString();
	}

	/**
	 * @see java.lang.Object#equals(Object)
	 */
	public boolean equals(Object object) {
		if (!(object instanceof Subject)) {
			return false;
		}
		Subject rhs = (Subject) object;
		return new EqualsBuilder().append(this.display,
				rhs.display).isEquals();
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return new HashCodeBuilder(1209070065, -1124530119).append(this.display).toHashCode();
	}

}
