/*
 * The SmartWeb Framework
 * Copyright (C) 2004-2006
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * For further informations on the SmartWeb Framework please visit
 *
 *                        http://smartweb.sourceforge.net
 */
package net.smartlab.web.auth;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Random;
import java.util.Set;

import net.smartlab.config.Configuration;
import net.smartlab.config.ConfigurationException;
import net.smartlab.config.Element;
import net.smartlab.web.BusinessException;
import net.smartlab.web.DAOException;
import net.smartlab.web.DataAccessObject.SearchInfo;
import net.smartlab.web.page.CollectionPaginator;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * TODO documentation
 * 
 * @author svetrini
 * @author rlogiacco
 */
public class Domain extends net.smartlab.web.Domain {

	

	/**
	 * Logger for this class
	 */
	private static final Log logger = LogFactory.getLog(Domain.class);

	/**
	 * Stores the singleton <code>instance</code> for this class.
	 * @uml.property  name="instance"
	 */
	private static Domain instance;

	/**
	 * @uml.property  name="sessions"
	 * @uml.associationEnd  multiplicity="(0 -1)" ordering="true" elementType="net.smartlab.web.auth.Permission" qualifier="securityToken:java.lang.String net.smartlab.web.auth.User"
	 */
	private static Map sessions = new HashMap();

	/**
	 * TODO documentation
	 * @uml.property  name="handlers"
	 */
	private Map handlers = new HashMap();

	/**
	 * TODO documentation
	 * @uml.property  name="scopes"
	 */
	private Map scopes = new HashMap();

	/**
	 * TODO documentation
	 * @uml.property  name="authenticator"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="net.smartlab.web.auth.AuthenticationHandler"
	 */
	private Collection authenticator = new ArrayList();

	/**
	 * TODO documentation
	 * @uml.property  name="authorizator"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="net.smartlab.web.auth.Domain"
	 */
	private Collection authorizator = new ArrayList();

	/**
	 * TODO documentation
	 * @uml.property  name="registrator"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="net.smartlab.web.auth.RegistrationHandler"
	 */
	private Collection registrator = new ArrayList();

	/**
	 * TODO documentation
	 * @uml.property  name="auditor"
	 * @uml.associationEnd  multiplicity="(0 -1)" elementType="net.smartlab.web.auth.Handler"
	 */
	private Collection auditor = new ArrayList();


	/**
	 * Blocks class instantiation to enforce the singleton pattern.
	 */
	private Domain() {
		super();
	}

	/**
	 * Returns the singleton instance.
	 * 
	 * @return returns the singleton instance.
	 * @uml.property name="instance"
	 */
	public static synchronized Domain getInstance() {
		if (instance == null) {
			instance = new Domain();
			instance.init();
			//testGuest();
		}
		return instance;
	}

//	/**
//	 * 
//	 */
//	private static void testGuest() {
//		try {
//			UserFactory.getInstance().findByKey(new Long(User.GUEST.getId()));
//		} catch (DAOException daoe) {
//			logger.error("Cannot find guest user", daoe);
//		}
//	}

	/**
	 * TODO documentation
	 */
	protected void init() {
		try {
			Configuration config = getConfiguration();
			if (config!=null){
				// configure generic handlers
				Iterator handlers = config.getElement("handlers").getElements().iterator();
				while (handlers.hasNext()) {
					Element handler = (Element)handlers.next();
					this.handlers.put(handler.getAttribute("id"), this.init(handler));
				}
				// configure global handlers
				Iterator globals = config.getElement("global").getElements().iterator();
				while (globals.hasNext()) {
					Element element = (Element)globals.next();
					if (element.getName().equals("authentication")) {
						authenticator.add((AuthenticationHandler)this.init(element));
					} else if (element.getName().equals("authorization")) {
						authorizator.add((AuthorizationHandler)this.init(element));
					} else if (element.getName().equals("registration")) {
						registrator.add((RegistrationHandler)this.init(element));
					} else if (element.getName().equals("audit")) {
						auditor.add((Handler)this.init(element));
					}
				}
			}
		} catch (Exception e) {
			logger.fatal("Initialization failure: ", e);
		}
	}




	/**
	 * @param handler
	 * @return
	 * @throws ConfigurationException
	 */
	private Handler init(Element handler) throws ConfigurationException {
		try {
			Handler instance = null;
			String type;
			try {
				type = handler.getAttribute("type");
			} catch (RuntimeException e) {
				type = null;
			}
			String refid;
			try {
				refid = handler.getAttribute("refid");
			} catch (RuntimeException e) {
				refid = null;
			}
			if (type != null) {
				instance = (Handler)Class.forName(handler.getAttribute("type")).newInstance();
				instance.setId(handler.getAttribute("id"));
				Map map = new HashMap();
				Iterator params = handler.getElements("param").iterator();
				while (params.hasNext()) {
					Element param = (Element)params.next();
					map.put(param.getAttribute("name"), param.getContent());
				}
				instance.init(map);
			} else if (refid != null) {
				instance = (Handler)this.handlers.get(handler.getAttribute("refid"));
			} else {
				throw new ConfigurationException("Error parsing configuration File");
			}
			return instance;
		} catch (Exception e) {
			logger.error("Initialization failure on " + handler.getAttribute("type"), e);
			return null;
		}
	}

	/**
	 * Allow users to self register in the system. The self registrator
	 * procedure could be customized through the
	 * <code>RegistrationStrategy</code> interface.
	 * 
	 * @param user the user to be registered
	 * @param props the map of properties to be providen to the custom
	 *        registrator process
	 * @param step TODO documentation
	 * @return
	 * @throws BusinessException if something wrong occurs during the
	 *         registrator process
	 */
	public String register(Map props, String step) throws BusinessException {
		try {
			for (Iterator iterator = registrator.iterator(); iterator.hasNext();) {
				RegistrationHandler reg = (RegistrationHandler)iterator.next();
				step = reg.onRegister(props, step);
			}
			return step;
		} catch (Exception e) {
			throw new BusinessException("Exception in registrator handling", e);
		}
	}

	/**
	 * Logs out a user from the system.
	 * 
	 * @param ticket the previously generated session ticket for the user.
	 * @throws BusinessException if something wrong occurs during the logout
	 *         procedure.
	 */
	public void logout(String securityToken) throws BusinessException {
		if (logger.isDebugEnabled()) {
			logger.debug("logout(securityToken=" + securityToken + ") - start");
		}
		if (securityToken != null && !"".equals(securityToken.trim())) {
			User user = (User)sessions.remove(securityToken);
			if (user != null) {
				user.logout();
				try {
					UserFactory.getInstance().update(user);
				} catch (DAOException daoe) {
					throw new AuthenticationException(daoe);
				}
			}
		}
		if (logger.isDebugEnabled()) {
			logger.debug("logout(securityToken=" + securityToken + ") - end");
		}
	}

	/**
	 * Logs in a user verifing the providen credentials matches the ones stored
	 * into the database.
	 * 
	 * @param credentials the user credentials which ensure the user is whoever
	 *        he pretends to impersonate.
	 * @return the authenticated user or <code>null</code> if authentication
	 *         fails.
	 * @throws BusinessException if something wrong occurs during credentials
	 *         verification.
	 */
	public User login(Credentials credentials) throws BusinessException {
		if (logger.isDebugEnabled()) {
			logger.debug("before login - user: "+User.get().getDisplay());
			logger.debug("login(credentials=" + credentials + ") - start");
		}		
		try {
			if (credentials==null || credentials.getUsername()==null || credentials.getUsername().trim()=="") {
				throw new AuthenticationException("Invalid credentials");
 			} 
			
			User user = (User)UserFactory.getInstance().findByUsername(credentials.getUsername());
			if (user != null) {
				try {
					for (Iterator iterator = authenticator.iterator(); iterator.hasNext();) {
						AuthenticationHandler authenticatorItem = (AuthenticationHandler)iterator.next();
						if (authenticatorItem!=null)
							authenticatorItem.onLogin(user, credentials);
					}
				} catch (AuthenticationException ae) {
					throw ae;
				} catch (Exception e) {
					throw new AuthenticationException(e);
				}
				String securityToken = Domain.generateSecurityToken();
				user.setSecurityToken(securityToken);
				logger.info("security token " + securityToken + " generated for user " + user.getId());
				Object previous = sessions.put(securityToken, user);
				if (previous != null) {
					throw new BusinessException("Security token collision detected: " + securityToken);
				}
				user.login();
				if (logger.isDebugEnabled()) {
					logger.debug("login(credentials=" + credentials + ") - end");
				}
				UserFactory.getInstance().update(user);
				logger.debug("end login - current user: "+User.get().getDisplay());
				logger.debug("end login - returned user: "+user.getDisplay());
				return user;
			} else {
				throw new AuthenticationException("Invalid credentials");
			}
		} catch (DAOException daoe) {
			throw new AuthenticationException(daoe);
		}
		
	}

	/**
	 * @param securityToken
	 * @param role
	 * @param scope
	 * @param params
	 * @return
	 * @throws AuthorizationException
	 */
	public boolean authorize(String securityToken, Role role, Scope scope, Map params) throws AuthorizationException {
		User user = (User)sessions.get(securityToken);
		if (user == null) {
			throw new AuthorizationException("Invalid user");
		}
		Set privilegesFromUser = role.getPrivileges();
		for (Iterator iterator = privilegesFromUser.iterator(); iterator.hasNext();) {
			Privilege privFromUser = (Privilege)iterator.next();
			if (!user.hasPrivilege(privFromUser, scope))
				return false;
		}
		return true;
	}

//	/**
//	 * @param securityToken
//	 * @param role
//	 * @param scope
//	 * @param params
//	 * @return
//	 * @throws AuthorizationException
//	 */
//	public boolean authorize(User user, Privilege privilege, BusinessObject scope, String[] handlersId, Map params)
//			throws AuthorizationException {
//		if (user != null && user.getSecurityToken() != null && Status.ENABLED.equals(user.getStatus())
//				&& sessions.get(user.getSecurityToken()).equals(user)) {
//			Scope sc = null;
//			if (scope != null) {
//				try {
//					sc = Scope.generate(scope);
//				} catch (BusinessException e) {
//					throw new AuthorizationException(e);
//				}
//			}
//			return user.hasPrivilege(privilege, sc);
//		}
//		// FIXME manage handlers
//		return false;
//	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @return
	 * @throws BusinessException
	 */
	public User findUser(String id) throws BusinessException {
		try {
			User user = (User)UserFactory.getInstance().findByKey(Long.valueOf(id));
			user.getGroups();
			user.getPolicy();
			return user;
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param info
	 * @return
	 * @throws BusinessException
	 */
	public Collection listUsers(SearchInfo info) throws BusinessException {
		try {
			return UserFactory.getInstance().list(info);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param info
	 * @return
	 * @throws BusinessException
	 */
	public Collection pageUsers(SearchInfo info) throws BusinessException {
		try {
			return UserFactory.getInstance().page(info);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param idSession
	 * @throws BusinessException
	 */
	public void removeUser(String idSession) throws BusinessException {
		try {
			UserFactory.getInstance().remove(this.findUser(idSession));
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param user
	 * @throws BusinessException
	 */
	public void updateUser(User user) throws BusinessException {
		try {
			UserFactory.getInstance().update(user);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param status
	 * @return
	 * @throws BusinessException
	 */
	public long count(User.Status status) throws BusinessException {
		try {
			return UserFactory.getInstance().count(status);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @return
	 * @throws BusinessException
	 */
	public Role findRole(String id) throws BusinessException {
		try {
			return (Role)RoleFactory.getInstance().findByKey(id);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param info
	 * @return
	 * @throws BusinessException
	 */
	public Collection listRoles(SearchInfo info) throws BusinessException {
		try {
			return RoleFactory.getInstance().list(info);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param info
	 * @return
	 * @throws BusinessException
	 */
	public Collection pageRoles(SearchInfo info) throws BusinessException {
		try {
			return new CollectionPaginator(RoleFactory.getInstance().list(info));
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @throws BusinessException
	 */
	public void removeRole(String id) throws BusinessException {
		try {
			RoleFactory.getInstance().remove(this.findRole(id));
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param role
	 * @throws BusinessException
	 */
	public void updateRole(Role role) throws BusinessException {
		try {
			RoleFactory.getInstance().update(role);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @return
	 * @throws BusinessException
	 */
	public Group findGroup(String id) throws BusinessException {
		try {
			return (Group)GroupFactory.getInstance().findByKey(id);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param info
	 * @return
	 * @throws BusinessException
	 */
	public Collection listGroups(SearchInfo info) throws BusinessException {
		try {
			return GroupFactory.getInstance().list(info);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param info
	 * @return
	 * @throws BusinessException
	 */
	public Collection pageGroups(SearchInfo info) throws BusinessException {
		try {
			return GroupFactory.getInstance().page(info);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @throws BusinessException
	 */
	public void removeGroup(String id) throws BusinessException {
		try {
			GroupFactory.getInstance().remove(this.findGroup(id));
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param group
	 * @throws BusinessException
	 */
	public void updateGroup(Group group) throws BusinessException {
		try {
			GroupFactory.getInstance().update(group);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param subject
	 * @param scope
	 * @param role
	 * @param handler
	 * @throws BusinessException
	 */
	public void registerPermission(Subject subject, Scope scope, Role role, AuthorizationHandler handler)
			throws BusinessException {
		Permission permission = new Permission();
		permission.setSubject(subject);
		permission.setScope(scope);
		permission.setRoleId(role.getId());
		PermissionFactory factory = PermissionFactory.getInstance();
		try {
			factory.update(permission);
		} catch (DAOException e) {
			throw new BusinessException("Exception durin permission saving: " + e.getMessage());
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @return
	 * @throws BusinessException
	 */
	public Permission findPermission(String id) throws BusinessException {
		try {
			return (Permission)PermissionFactory.getInstance().findByKey(id);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @throws BusinessException
	 */
	public void removePermission(String id) throws BusinessException {
		try {
			PermissionFactory.getInstance().remove(this.findPermission(id));
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param info
	 * @return
	 * @throws BusinessException
	 */
	public Collection listPermissions(SearchInfo info) throws BusinessException {
		try {
			return GroupFactory.getInstance().list(info);
		} catch (DAOException daoe) {
			throw new BusinessException(daoe);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @return
	 * @throws BusinessException
	 */
	public Subject findSubject(String id) throws BusinessException {
		User user = null;
		Group group = null;
		try {
			group = (Group)GroupFactory.getInstance().findByKey(id);
			if (group != null)
				return group;
		} catch (DAOException daoe) {
			// nope
		}
		try {
			user = (User)UserFactory.getInstance().findByKey(id);
			if (user != null)
				return user;
		} catch (DAOException daoe) {
			// nope
		}
		return null;
	}

	/**
	 * TODO documentation
	 * 
	 * @param permission
	 * @throws BusinessException
	 */
	public void updatePermission(Permission permission) throws BusinessException {
		try {
			PermissionFactory.getInstance().update(permission);
		} catch (DAOException e) {
			throw new BusinessException("error saving permission: " + permission, e);
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @return
	 */
	public Map getScopeTypes() {
		return scopes;
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @return
	 */
	protected AuthenticationHandler getAuthenticationHandler(String id) {
		try {
			return (AuthenticationHandler)handlers.get(id);
		} catch (ClassCastException cce) {
			return null;
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @return
	 */
	protected AuthorizationHandler getAuthorizationHandler(String id) {
		try {
			return (AuthorizationHandler)handlers.get(id);
		} catch (ClassCastException cce) {
			return null;
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param id
	 * @return
	 */
	protected AuditHandler getAuditHandler(String id) {
		try {
			return (AuditHandler)handlers.get(id);
		} catch (ClassCastException cce) {
			return null;
		}
	}

	/**
	 * TODO documentation
	 * 
	 * @param node
	 * @return
	 */
	public Handler getHandler(String node) {
		return (Handler)this.handlers.get(node);
	}

	/**
	 * TODO documentation
	 * 
	 * @return
	 */
	private final static String generateSecurityToken() {
		Random random = new Random();
		String token = Long.toHexString(random.nextLong());
		return token;
	}
	
	/**
	 * Set ThreadLocal with current User from  a SecurityToken
	 * @param securityToken
	 * @throws AuthenticationException
	 */
	protected final static void setUser(String securityToken) throws AuthenticationException{
		User user = null;
		if (User.GUEST.getSecurityToken().equals(securityToken)){
			user = User.GUEST;
		}else{
			user = (User) sessions.get(securityToken);
		}		
		if (user!=null){
			User.set(user);
		}else{
			throw new AuthenticationException("Invalid security Token");
		}
		
	}
	
}
