/*
 * The SmartWeb Framework
 * Copyright (C) 2004-2006
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * For further informations on the SmartWeb Framework please visit
 *
 *                        http://smartweb.sourceforge.net
 */
package net.smartlab.web.auth;

import java.io.Serializable;

/**
 * The username and secret pair this class represents are the informations
 * needed to uniquely identify a user into the system and verify the user is who
 * he pretends to be.
 * 
 * @author rlogiacco
 * @author svetrini
 * @author pmoretti
 */
public class Credentials implements Serializable {

	private static final long serialVersionUID = -5314053151889204398L;

	/**
	 * Character used to separate the username and it's secret when represented
	 * as a <code>String</code>.
	 */
	public final static char SEPARATOR = '\r';

	/**
	 * The alphanumeric sequence used to identify the user into the system.
	 * 
	 * @uml.property name="username"
	 */
	private String username;

	/**
	 * The binary sequence used to authenticate the user.
	 * 
	 * @uml.property name="secret"
	 */
	private byte[] secret;


	/**
	 * Creates a new empty instance.
	 */
	public Credentials() {
		super();
	}

	/**
	 * Creates an instance from a <code>String</code> representation of the
	 * credentials.
	 * 
	 * @param value
	 */
	public Credentials(String value) {
		int split = value.indexOf(SEPARATOR);
		this.username = value.substring(0, split);
		this.secret = value.substring(split + 1).getBytes();
	}

	/**
	 * Returns the alphanumeric sequence used to identify the user into the
	 * system.
	 * 
	 * @return the unique name used for authentication.
	 * @hibernate.property column="`username`" length="50" not-null="true"
	 *                     unique="true"
	 * @uml.property name="username"
	 */
	public String getUsername() {
		return this.username;
	}

	/**
	 * Sets the alphanumeric sequence used to identify the user into the system.
	 * This method will fail to set the value if it already exists to
	 * enforce the uniqueness of this attribute.
	 * 
	 * @param username the alphanumeric sequence used to identify the user into
	 *        the system.
	 * @uml.property name="username"
	 */
	public void setUsername(String username) {
		if (this.username == null || !(this.username.equals(username))) {
			this.username = username;
		}
	}

	/**
	 * Returns the binary sequence used to authenticate the user.
	 * 
	 * @return the binary sequence used to authenticate the user.
	 * @hibernate.property column="`secret`" length="255"
	 * @uml.property name="secret"
	 */
	public byte[] getSecret() {
		return secret;
	}

	/**
	 * Sets the binary sequence used to authenticate the user.
	 * 
	 * @param secret the binary sequence used to authenticate the user.
	 * @uml.property name="secret"
	 */
	public void setSecret(byte[] secret) {
		this.secret = secret;
	}
	
	/**
	 * Sets the binary sequence used to authenticate the user.
	 * 
	 * @param secret the binary sequence used to authenticate the user.
	 * @uml.property name="secret"
	 */
	public void setSecret(String secret) {
		if(secret!=null)
			this.secret = secret.getBytes();
		else
			this.secret = new byte[0];			
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		String strSecret=null;
		String result = "";
		if (secret!=null) strSecret = new String(secret);
		result = result +  username + SEPARATOR + strSecret; 
		return result;
	}
}
