/* LICENSE */

package net.smartlab.config;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.xml.sax.Attributes;

/**
 * @testcase net.smartlab.config.test.TestNode
 */
public class Node extends Element {

	/**
	 * The node sub elements.
	 *
	 * @link aggregation
	 * @associates <{Element}>
	 */
	protected List children;

	/**
	 * The node attributes.
	 *
	 * @link aggregation
	 * @associates <{Attribute}>
	 */
	protected Map attributes;

	/**
	 * The node direct contents.
	 */
	protected String content;

	/**
	 * Marks this node as already resolved for references.
	 */
	protected boolean resolved;

	/**
	 * Constructs a generic empty node.
	 *
	 * @param parent the node containing this node.
	 * @param name a name associated to this node.
	 * @param attributes a set of attributes associated with this node.
	 */
	protected Node(Node parent, String name, Attributes attributes) {
		super(parent, name);
		this.children = new ArrayList();
		if (attributes != null) {
			this.attributes = new HashMap(attributes.getLength());
			for (int i = 0; i < attributes.getLength(); i++) {
				this.attributes.put(attributes.getQName(i), attributes.getValue(i));
			}
		}
	}

	/**
	 * @see net.smartlab.config.Element#getId()
	 */
	public String getId() {
		return (String)attributes.get(Reference.IDENTIFIER);
	}

	/**
	 * @see net.smartlab.config.Element#getElements()
	 */
	public Collection getElements() {
		return Collections.unmodifiableCollection(children);
	}

	/**
	 * @see net.smartlab.config.Element#getElements(java.lang.String)
	 */
	public Collection getElements(String name) {
		List result = new ArrayList();
		Iterator children = this.children.iterator();
		while (children.hasNext()) {
			Element child = (Element)children.next();
			if (name.equals(child.getName())) {
				result.add(child);
			}
		}
		return Collections.unmodifiableCollection(result);
	}

	/**
	 * @see net.smartlab.config.Element#getElement(java.lang.String,
	 *      java.lang.String, java.lang.String)
	 */
	public Element getElement(String name, String attribute, String value) throws ConfigurationException {
		Iterator children = this.children.iterator();
		while (children.hasNext()) {
			Element child = (Element)children.next();
			if (name.equals(child.name)) {
				if (attribute != null) {
					String found = child.getAttribute(attribute);
					if (found != null) {
						if (value != null) {
							if (value.equals(found)) {
								return child;
							}
						} else {
							return child;
						}
					}
				} else {
					return child;
				}
			}
		}
		return null;
	}

	/**
	 * @see net.smartlab.config.Element#getAttributeNames()
	 */
	public Collection getAttributeNames() {
		return Collections.unmodifiableCollection(attributes.entrySet());
	}

	/**
	 * @see net.smartlab.config.Element#getAttribute(java.lang.String)
	 */
	public String getAttribute(String name) {
		return (String)attributes.get(name);
	}

	/**
	 * @see net.smartlab.config.Element#getContent()
	 */
	public String getContent() {
		return content;
	}

	/**
	 * @see net.smartlab.config.Element#resolve()
	 */
	public void resolve() throws ConfigurationException {
		if (!resolved) {
			Iterator elements = children.iterator();
			while (elements.hasNext()) {
				((Element)elements.next()).resolve();
			}
			resolved = true;
		}
	}

	/**
	 * Resolves the requested reference through its children nodes or,
	 * recursively in the parent node children.
	 *
	 * @param name the referenced element name.
	 * @param id the referenced element identifier.
	 * @return the referenced element.
	 * @throws ConfigurationException if the referenced element cannot be found
	 *             in the wntire configuration tree.
	 */
	protected Node resolve(String name, String id) throws ConfigurationException {
		Iterator children = this.children.iterator();
		while (children.hasNext()) {
			Element child = (Element)children.next();
			if (child instanceof Node && name.equals(child.name) && id.equals(child.getId())) {
				return (Node)child;
			}
		}
		return parent.resolve(name, id);
	}
}
