/* LICENSE */

package net.smartlab.config;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.StringTokenizer;

/**
 * This class identifies a configuration element which can in turn contains
 * other configuration elements. An element is composed by a name, a set of
 * attributes which can be empty and, optionally, by its contents.
 *
 * @author rlogiacco@smartlab.net
 */
abstract public class Element {

	/**
	 * The containing node.
	 */
	protected Node parent;

	/**
	 * Te element name.
	 */
	protected String name;

	/**
	 * Constructs an element with a givenparent node and name.
	 *
	 * @param parent the node containing the element.
	 * @param name the element name.
	 */
	protected Element(Node parent, String name) {
		this.parent = parent;
		this.name = name;
	}

	/**
	 * Returns the name of this configuration element.
	 *
	 * @return a <code>String</code> representing the name of this
	 *         configuration element.
	 */
	public String getName() {
		return name;
	}

	/**
	 * Returns the unique identifier associated with this element.
	 *
	 * @return the unique identifier associated with this element.
	 */
	public abstract String getId();

	/**
	 * Returns all the contained elements.
	 *
	 * @return an unmodifiable <code>Collection</code> which enumerates the
	 *         contained elements.
	 * @throws ConfigurationException if something wrong occurs during the
	 *             operation.
	 */
	public abstract Collection getElements() throws ConfigurationException;

	/**
	 * Returns all the contained elements which name equals the given one.
	 *
	 * @param name the name of the elements to search for.
	 * @return an unmodifiable <code>Collection</code> which enumerates the
	 *         contained elements which name equals the given one.
	 * @throws ConfigurationException if something wrong occurs during the
	 *             operation.
	 */
	public abstract Collection getElements(String name) throws ConfigurationException;

	/**
	 * Returns the first contained element whose name equals the given one.
	 *
	 * @param name the name of the element to search for.
	 * @return the contained <code>Element</code> or <code>null</code> if
	 *         the search fails.
	 * @throws ConfigurationException if something wrong occurs during the
	 *             operation.
	 */
	public Element getElement(String name) throws ConfigurationException {
		return this.getElement(name, null, null);
	}

	/**
	 * Returns the first contained element whose name equals the given one and
	 * whose attribute list contains the given one.
	 *
	 * @param name the name of the element to search for.
	 * @param attribute the name of the attribute which must be defined on the
	 *            element.
	 * @return the contained <code>Element</code> or <code>null</code> if
	 *         the search fails.
	 * @throws ConfigurationException if something wrong occurs during the
	 *             operation.
	 */
	public Element getElement(String name, String attribute) throws ConfigurationException {
		return this.getElement(name, attribute, null);
	}

	/**
	 * Returns the first contained element whose name equals the given one and
	 * whose attribute list defines an attribute whose name and values matches
	 * the given parameters.
	 *
	 * @param name the name of the element to search for.
	 * @param attribute attribute the name of the attribute which must be
	 *            defined on the element.
	 * @param value the value the attribute must match.
	 * @return the contained <code>Element</code> or <code>null</code> if
	 *         the search fails.
	 * @throws ConfigurationException if something wrong occurs during the
	 *             operation.
	 */
	public abstract Element getElement(String name, String attribute, String value) throws ConfigurationException;

	/**
	 * Returns all the attributes of this element.
	 *
	 * @return an unmodifiable <code>Collection</code> which enumerates the
	 *         attributes of this element.
	 * @throws ConfigurationException if something wrong occurs during the
	 *             operation.
	 */
	public abstract Collection getAttributeNames() throws ConfigurationException;

	/**
	 * Returns the value of the attribute whose name equals the specified one or
	 * <code>null</code> if the element doesn't contain a such named
	 * attribute.
	 *
	 * @param name the name of the attribute to get.
	 * @return a <code>String</code> instance representing the attribute value
	 *         or <code>null</code> if no attribute with the specified name
	 *         was found.
	 * @throws ConfigurationException if something wrong occurs during the
	 *             operation.
	 */
	public abstract String getAttribute(String name) throws ConfigurationException;

	/**
	 * Returns an unmodifiable <code>Collection</code> build on the attribute
	 * value parsed as a comma separated list.
	 *
	 * @param name the name of the attribute whose value will be parsed as a
	 *            comma separated list.
	 * @return an unmodifiable <code>Collection</code> which enumerates the
	 *         attribute values parsed as a comma separated list.
	 * @throws ConfigurationException if something wrong occurs during the
	 *             operation.
	 */
	public Collection getAttributeValues(String name) throws ConfigurationException {
		Collection list = new ArrayList();
		StringTokenizer tokenizer = new StringTokenizer(this.getAttribute(name), ",");
		while (tokenizer.hasMoreTokens()) {
			list.add(tokenizer.nextToken());
		}
		return list;
	}

	/**
	 * Returns the element content as a string.
	 *
	 * @return the characters contained between the tag boundaries.
	 * @throws ConfigurationException if the content couldn't be retrieved.
	 */
	public abstract String getContent() throws ConfigurationException;

	/**
	 * Resolves the element reference.
	 *
	 * @throws ConfigurationException if something wrong occurs during the
	 *             operation.
	 */
	public abstract void resolve() throws ConfigurationException;

	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		try {
			StringBuffer result = new StringBuffer();
			result.append('<');
			result.append(this.name);
			Iterator attributes = this.getAttributeNames().iterator();
			while (attributes.hasNext()) {
				String name = (String)attributes.next();
				result.append(' ');
				result.append(name);
				result.append("='");
				result.append(this.getAttribute(name));
				result.append("'");
			}
			result.append(">\n");
			result.append(this.getContent());
			Iterator elements = this.getElements().iterator();
			while (elements.hasNext()) {
				result.append(elements.next().toString());
			}
			result.append("</");
			result.append(this.name);
			result.append(">\n");
			return result.toString();
		} catch (ConfigurationException ce) {
			return ce.toString();
		}
	}
}
